/**
 * Property-Based Tests for Form Logic
 * 
 * **Feature: student-registration-form**
 * 
 * Uses fast-check for property-based testing
 */

const fc = require('fast-check');
const {
    shouldShowStudyField,
    handleCityChange,
    setLoadingState,
    isAnyLoading,
    handleSubmissionError,
    validatePhoneNumber,
    validateForm,
    createInitialState
} = require('./form-logic');

describe('Form Properties', () => {
    /**
     * **Feature: student-registration-form, Property 4: Study Field Visibility Based on Grade Level**
     * **Validates: Requirements 2.2, 2.3**
     * 
     * *For any* grade level selection, the study field input should be visible 
     * if and only if the grade level is 10, 11, or 12.
     */
    describe('Property 4: Study Field Visibility Based on Grade Level', () => {
        test('study field should be visible only for grades 10, 11, 12', () => {
            fc.assert(
                fc.property(
                    fc.integer({ min: 1, max: 12 }),
                    (gradeLevel) => {
                        const shouldShow = shouldShowStudyField(gradeLevel);
                        const expectedVisible = gradeLevel >= 10 && gradeLevel <= 12;
                        
                        return shouldShow === expectedVisible;
                    }
                ),
                { numRuns: 100 }
            );
        });

        test('study field should be hidden for null/undefined grade level', () => {
            fc.assert(
                fc.property(
                    fc.constantFrom(null, undefined),
                    (gradeLevel) => {
                        return shouldShowStudyField(gradeLevel) === false;
                    }
                ),
                { numRuns: 10 }
            );
        });

        test('study field visibility is consistent across multiple checks', () => {
            fc.assert(
                fc.property(
                    fc.integer({ min: 1, max: 12 }),
                    (gradeLevel) => {
                        // Multiple calls should return same result
                        const result1 = shouldShowStudyField(gradeLevel);
                        const result2 = shouldShowStudyField(gradeLevel);
                        const result3 = shouldShowStudyField(gradeLevel);
                        
                        return result1 === result2 && result2 === result3;
                    }
                ),
                { numRuns: 100 }
            );
        });
    });

    /**
     * **Feature: student-registration-form, Property 2: City Change Resets Branch Selection**
     * **Validates: Requirements 1.3**
     * 
     * *For any* form state with a selected branch, when the city selection changes,
     * the branch selection should become null and the new branches should be loaded for the new city.
     */
    describe('Property 2: City Change Resets Branch Selection', () => {
        test('city change should reset branch, gradeLevel, studyField, and school', () => {
            fc.assert(
                fc.property(
                    // Generate a state with some selections
                    fc.record({
                        cityId: fc.integer({ min: 1, max: 100 }),
                        branchId: fc.integer({ min: 1, max: 100 }),
                        gradeLevel: fc.integer({ min: 1, max: 12 }),
                        studyField: fc.constantFrom('sayisal', 'sozel', 'esit_agirlik', 'dil', null),
                        schoolId: fc.integer({ min: 1, max: 1000 }),
                        schoolName: fc.string({ minLength: 1, maxLength: 100 }),
                        studentFirstName: fc.string(),
                        studentLastName: fc.string(),
                        parentFirstName: fc.string(),
                        parentLastName: fc.string(),
                        parentPhone1: fc.string(),
                        parentPhone2: fc.string(),
                        policyAccepted: fc.boolean()
                    }),
                    fc.integer({ min: 1, max: 100 }), // New city ID
                    (currentState, newCityId) => {
                        const newState = handleCityChange(currentState, newCityId);
                        
                        // City should be updated
                        if (newState.cityId !== newCityId) return false;
                        
                        // Dependent fields should be reset
                        if (newState.branchId !== null) return false;
                        if (newState.gradeLevel !== null) return false;
                        if (newState.studyField !== null) return false;
                        if (newState.schoolId !== null) return false;
                        if (newState.schoolName !== '') return false;
                        
                        // Non-dependent fields should be preserved
                        if (newState.studentFirstName !== currentState.studentFirstName) return false;
                        if (newState.studentLastName !== currentState.studentLastName) return false;
                        if (newState.parentFirstName !== currentState.parentFirstName) return false;
                        if (newState.parentLastName !== currentState.parentLastName) return false;
                        if (newState.parentPhone1 !== currentState.parentPhone1) return false;
                        if (newState.parentPhone2 !== currentState.parentPhone2) return false;
                        if (newState.policyAccepted !== currentState.policyAccepted) return false;
                        
                        return true;
                    }
                ),
                { numRuns: 100 }
            );
        });

        test('city change to null should also reset dependent fields', () => {
            fc.assert(
                fc.property(
                    fc.record({
                        cityId: fc.integer({ min: 1, max: 100 }),
                        branchId: fc.integer({ min: 1, max: 100 }),
                        gradeLevel: fc.integer({ min: 1, max: 12 }),
                        studyField: fc.constantFrom('sayisal', 'sozel', null),
                        schoolId: fc.integer({ min: 1, max: 1000 }),
                        schoolName: fc.string(),
                        studentFirstName: fc.string(),
                        studentLastName: fc.string(),
                        parentFirstName: fc.string(),
                        parentLastName: fc.string(),
                        parentPhone1: fc.string(),
                        parentPhone2: fc.string(),
                        policyAccepted: fc.boolean()
                    }),
                    (currentState) => {
                        const newState = handleCityChange(currentState, null);
                        
                        return newState.cityId === null &&
                               newState.branchId === null &&
                               newState.gradeLevel === null &&
                               newState.studyField === null &&
                               newState.schoolId === null;
                    }
                ),
                { numRuns: 100 }
            );
        });
    });

    /**
     * **Feature: student-registration-form, Property 13: Loading State During Async Operations**
     * **Validates: Requirements 6.4**
     * 
     * *For any* async operation (API call), the loading state should be true during 
     * the operation and false after completion (success or error).
     */
    describe('Property 13: Loading State During Async Operations', () => {
        test('setting loading state should update only the specified operation', () => {
            fc.assert(
                fc.property(
                    fc.constantFrom('cities', 'branches', 'gradeLevels', 'schools', 'submit'),
                    fc.boolean(),
                    (operation, isLoading) => {
                        const initialState = {
                            cities: false,
                            branches: false,
                            gradeLevels: false,
                            schools: false,
                            submit: false
                        };
                        
                        const newState = setLoadingState(initialState, operation, isLoading);
                        
                        // The specified operation should have the new value
                        if (newState[operation] !== isLoading) return false;
                        
                        // Other operations should remain unchanged
                        for (const key of Object.keys(initialState)) {
                            if (key !== operation && newState[key] !== initialState[key]) {
                                return false;
                            }
                        }
                        
                        return true;
                    }
                ),
                { numRuns: 100 }
            );
        });

        test('isAnyLoading should return true if any operation is loading', () => {
            fc.assert(
                fc.property(
                    fc.record({
                        cities: fc.boolean(),
                        branches: fc.boolean(),
                        gradeLevels: fc.boolean(),
                        schools: fc.boolean(),
                        submit: fc.boolean()
                    }),
                    (loadingState) => {
                        const anyLoading = isAnyLoading(loadingState);
                        const expectedAnyLoading = Object.values(loadingState).some(v => v === true);
                        
                        return anyLoading === expectedAnyLoading;
                    }
                ),
                { numRuns: 100 }
            );
        });

        test('loading state transitions should be consistent', () => {
            fc.assert(
                fc.property(
                    fc.constantFrom('cities', 'branches', 'gradeLevels', 'schools', 'submit'),
                    (operation) => {
                        const initialState = {
                            cities: false,
                            branches: false,
                            gradeLevels: false,
                            schools: false,
                            submit: false
                        };
                        
                        // Start loading
                        const loadingState = setLoadingState(initialState, operation, true);
                        
                        // Should be loading
                        if (!loadingState[operation]) return false;
                        if (!isAnyLoading(loadingState)) return false;
                        
                        // Stop loading
                        const doneState = setLoadingState(loadingState, operation, false);
                        
                        // Should not be loading
                        if (doneState[operation]) return false;
                        
                        return true;
                    }
                ),
                { numRuns: 100 }
            );
        });
    });

    /**
     * **Feature: student-registration-form, Property 14: Form State Preservation on Error**
     * **Validates: Requirements 7.2**
     * 
     * *For any* form submission that results in an error, all previously entered 
     * form data should remain intact after the error is displayed.
     */
    describe('Property 14: Form State Preservation on Error', () => {
        test('form data should be preserved when submission error occurs', () => {
            fc.assert(
                fc.property(
                    // Generate form state with data
                    fc.record({
                        cityId: fc.integer({ min: 1, max: 100 }),
                        branchId: fc.integer({ min: 1, max: 100 }),
                        gradeLevel: fc.integer({ min: 1, max: 12 }),
                        studyField: fc.constantFrom('sayisal', 'sozel', 'esit_agirlik', 'dil', null),
                        studentFirstName: fc.string({ minLength: 1, maxLength: 50 }),
                        studentLastName: fc.string({ minLength: 1, maxLength: 50 }),
                        schoolId: fc.integer({ min: 1, max: 1000 }),
                        schoolName: fc.string({ minLength: 1, maxLength: 100 }),
                        parentFirstName: fc.string({ minLength: 1, maxLength: 50 }),
                        parentLastName: fc.string({ minLength: 1, maxLength: 50 }),
                        parentPhone1: fc.string({ minLength: 10, maxLength: 15 }),
                        parentPhone2: fc.string({ minLength: 0, maxLength: 15 }),
                        policyAccepted: fc.boolean()
                    }),
                    // Generate error
                    fc.record({
                        message: fc.string({ minLength: 1, maxLength: 100 }),
                        errors: fc.dictionary(
                            fc.constantFrom('city', 'branch', 'gradeLevel', 'studentFirstName'),
                            fc.string({ minLength: 1, maxLength: 50 })
                        )
                    }),
                    (formState, error) => {
                        const stateAfterError = handleSubmissionError(formState, error);
                        
                        // All form data should be preserved
                        if (stateAfterError.cityId !== formState.cityId) return false;
                        if (stateAfterError.branchId !== formState.branchId) return false;
                        if (stateAfterError.gradeLevel !== formState.gradeLevel) return false;
                        if (stateAfterError.studyField !== formState.studyField) return false;
                        if (stateAfterError.studentFirstName !== formState.studentFirstName) return false;
                        if (stateAfterError.studentLastName !== formState.studentLastName) return false;
                        if (stateAfterError.schoolId !== formState.schoolId) return false;
                        if (stateAfterError.schoolName !== formState.schoolName) return false;
                        if (stateAfterError.parentFirstName !== formState.parentFirstName) return false;
                        if (stateAfterError.parentLastName !== formState.parentLastName) return false;
                        if (stateAfterError.parentPhone1 !== formState.parentPhone1) return false;
                        if (stateAfterError.parentPhone2 !== formState.parentPhone2) return false;
                        if (stateAfterError.policyAccepted !== formState.policyAccepted) return false;
                        
                        // Error information should be added
                        if (!stateAfterError.errors) return false;
                        if (!stateAfterError.submitError) return false;
                        
                        return true;
                    }
                ),
                { numRuns: 100 }
            );
        });

        test('multiple errors should not corrupt form state', () => {
            fc.assert(
                fc.property(
                    fc.record({
                        cityId: fc.integer({ min: 1, max: 100 }),
                        branchId: fc.integer({ min: 1, max: 100 }),
                        gradeLevel: fc.integer({ min: 1, max: 12 }),
                        studyField: fc.constantFrom('sayisal', 'sozel', null),
                        studentFirstName: fc.string(),
                        studentLastName: fc.string(),
                        schoolId: fc.integer({ min: 1, max: 1000 }),
                        schoolName: fc.string(),
                        parentFirstName: fc.string(),
                        parentLastName: fc.string(),
                        parentPhone1: fc.string(),
                        parentPhone2: fc.string(),
                        policyAccepted: fc.boolean()
                    }),
                    fc.array(
                        fc.record({
                            message: fc.string(),
                            errors: fc.dictionary(fc.string(), fc.string())
                        }),
                        { minLength: 1, maxLength: 5 }
                    ),
                    (formState, errors) => {
                        let currentState = formState;
                        
                        // Apply multiple errors
                        for (const error of errors) {
                            currentState = handleSubmissionError(currentState, error);
                        }
                        
                        // Original form data should still be preserved
                        return currentState.cityId === formState.cityId &&
                               currentState.branchId === formState.branchId &&
                               currentState.gradeLevel === formState.gradeLevel &&
                               currentState.studentFirstName === formState.studentFirstName &&
                               currentState.studentLastName === formState.studentLastName;
                    }
                ),
                { numRuns: 100 }
            );
        });
    });
});
