/**
 * Form Logic Module for Testing
 * 
 * Extracted logic from frontend modules for property-based testing
 */

/**
 * Determine if study field should be visible based on grade level
 * Property 4: Study Field Visibility Based on Grade Level
 * Requirements: 2.2, 2.3
 * 
 * @param {number|null} gradeLevel - The selected grade level (1-12)
 * @returns {boolean} - Whether study field should be visible
 */
function shouldShowStudyField(gradeLevel) {
    if (gradeLevel === null || gradeLevel === undefined) {
        return false;
    }
    return gradeLevel >= 10 && gradeLevel <= 12;
}

/**
 * Simulate city change and return new form state
 * Property 2: City Change Resets Branch Selection
 * Requirements: 1.3
 * 
 * @param {Object} currentState - Current form state
 * @param {number|null} newCityId - New city ID
 * @returns {Object} - New form state after city change
 */
function handleCityChange(currentState, newCityId) {
    return {
        ...currentState,
        cityId: newCityId,
        // Reset dependent fields
        branchId: null,
        gradeLevel: null,
        studyField: null,
        schoolId: null,
        schoolName: ''
    };
}

/**
 * Simulate loading state management
 * Property 13: Loading State During Async Operations
 * Requirements: 6.4
 * 
 * @param {Object} loadingState - Current loading state
 * @param {string} operation - Operation type
 * @param {boolean} isLoading - Whether operation is loading
 * @returns {Object} - New loading state
 */
function setLoadingState(loadingState, operation, isLoading) {
    return {
        ...loadingState,
        [operation]: isLoading
    };
}

/**
 * Check if any loading operation is in progress
 * Property 13: Loading State During Async Operations
 * 
 * @param {Object} loadingState - Loading state object
 * @returns {boolean} - Whether any operation is loading
 */
function isAnyLoading(loadingState) {
    return Object.values(loadingState).some(v => v === true);
}

/**
 * Simulate form submission error handling
 * Property 14: Form State Preservation on Error
 * Requirements: 7.2
 * 
 * @param {Object} formState - Current form state
 * @param {Object} error - Error object
 * @returns {Object} - Form state after error (should be preserved)
 */
function handleSubmissionError(formState, error) {
    // Form state should be preserved on error
    return {
        ...formState,
        errors: error.errors || {},
        submitError: error.message || 'Bir hata oluştu'
    };
}

/**
 * Validate phone number format (Turkish mobile)
 * Property 8: Phone Number Format Validation
 * Requirements: 4.2, 4.4
 * 
 * @param {string} phone - Phone number to validate
 * @returns {boolean} - Whether phone is valid
 */
function validatePhoneNumber(phone) {
    if (!phone) return false;
    const normalized = phone.replace(/\s+/g, '').replace(/[^0-9]/g, '');
    return /^5[0-9]{9}$/.test(normalized);
}

/**
 * Validate form state
 * Property 6: Required Field Validation
 * 
 * @param {Object} state - Form state
 * @returns {{isValid: boolean, errors: Object}}
 */
function validateForm(state) {
    const errors = {};
    
    if (!state.cityId) errors.city = 'Şehir alanı zorunludur';
    if (!state.branchId) errors.branch = 'Şube alanı zorunludur';
    if (!state.gradeLevel) errors.gradeLevel = 'Sınıf düzeyi alanı zorunludur';
    
    // Conditional study field validation (Property 7)
    if (shouldShowStudyField(state.gradeLevel) && !state.studyField) {
        errors.studyField = 'Alan bilgisi alanı zorunludur';
    }
    
    if (!state.studentFirstName?.trim()) errors.studentFirstName = 'Öğrenci adı alanı zorunludur';
    if (!state.studentLastName?.trim()) errors.studentLastName = 'Öğrenci soyadı alanı zorunludur';
    if (!state.schoolId) errors.school = 'Okul alanı zorunludur';
    if (!state.parentFirstName?.trim()) errors.parentFirstName = 'Veli adı alanı zorunludur';
    if (!state.parentLastName?.trim()) errors.parentLastName = 'Veli soyadı alanı zorunludur';
    
    if (!state.parentPhone1?.trim()) {
        errors.parentPhone1 = 'Telefon numarası alanı zorunludur';
    } else if (!validatePhoneNumber(state.parentPhone1)) {
        errors.parentPhone1 = 'Geçerli bir telefon numarası giriniz';
    }
    
    // Optional phone2 validation (Property 9)
    if (state.parentPhone2?.trim() && !validatePhoneNumber(state.parentPhone2)) {
        errors.parentPhone2 = 'Geçerli bir telefon numarası giriniz';
    }
    
    if (!state.policyAccepted) errors.policy = 'Gizlilik politikasını kabul etmeniz gerekmektedir';
    
    return {
        isValid: Object.keys(errors).length === 0,
        errors
    };
}

/**
 * Create initial form state
 */
function createInitialState() {
    return {
        cityId: null,
        branchId: null,
        gradeLevel: null,
        studyField: null,
        studentFirstName: '',
        studentLastName: '',
        schoolId: null,
        schoolName: '',
        parentFirstName: '',
        parentLastName: '',
        parentPhone1: '',
        parentPhone2: '',
        policyAccepted: false,
        loading: {
            cities: false,
            branches: false,
            gradeLevels: false,
            schools: false,
            submit: false
        },
        errors: {}
    };
}

module.exports = {
    shouldShowStudyField,
    handleCityChange,
    setLoadingState,
    isAnyLoading,
    handleSubmissionError,
    validatePhoneNumber,
    validateForm,
    createInitialState
};
