<?php
/**
 * Property-Based Tests for ValidationService
 * 
 * **Feature: student-registration-form**
 */

declare(strict_types=1);

namespace Tests\Property;

use PHPUnit\Framework\TestCase;
use Eris\Generator;
use Eris\TestTrait;
use App\Services\ValidationService;
use App\Models\StudyField;

class ValidationServicePropertyTest extends TestCase
{
    use TestTrait;

    private ValidationService $validationService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->validationService = new ValidationService();
    }

    /**
     * **Feature: student-registration-form, Property 8: Phone Number Format Validation**
     * **Validates: Requirements 4.2, 4.4**
     * 
     * *For any* phone number input, the validation should accept only strings 
     * matching the Turkish mobile format (5XX XXX XX XX or 5XXXXXXXXX). 
     * Invalid formats should return a validation error.
     */
    public function testPhoneNumberFormatValidation(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                Generator\string(), // Random strings to test invalid formats
                $this->validPhoneGenerator() // Valid phone numbers
            )
            ->then(function (string $randomString, string $validPhone) {
                // Valid phones should pass validation
                $this->assertTrue(
                    $this->validationService->validatePhoneNumber($validPhone),
                    "Valid phone '{$validPhone}' should pass validation"
                );

                // Check if random string is a valid phone format
                $cleaned = preg_replace('/[^0-9]/', '', $randomString);
                $isValidFormat = preg_match('/^5[0-9]{9}$/', $cleaned) === 1;
                
                $this->assertEquals(
                    $isValidFormat,
                    $this->validationService->validatePhoneNumber($randomString),
                    "Phone validation result should match expected for '{$randomString}'"
                );
            });
    }

    /**
     * **Feature: student-registration-form, Property 12: Input Sanitization**
     * **Validates: Requirements 7.3**
     * 
     * *For any* string input containing potential XSS or SQL injection patterns,
     * the sanitization function should remove or escape dangerous characters 
     * while preserving the safe content.
     */
    public function testInputSanitization(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                Generator\string()
            )
            ->then(function (string $input) {
                $sanitized = $this->validationService->sanitizeInput($input);
                
                // Sanitized output should not contain unescaped HTML tags
                $this->assertStringNotContainsString('<script', strtolower($sanitized));
                $this->assertStringNotContainsString('</script', strtolower($sanitized));
                
                // If input had < or >, they should be escaped
                if (str_contains($input, '<')) {
                    $this->assertStringContainsString('&lt;', $sanitized);
                }
                if (str_contains($input, '>')) {
                    $this->assertStringContainsString('&gt;', $sanitized);
                }
                
                // Null bytes should be removed
                $this->assertStringNotContainsString("\0", $sanitized);
            });
    }

    /**
     * **Feature: student-registration-form, Property 6: Required Field Validation**
     * **Validates: Requirements 1.4, 3.3, 3.4, 4.3, 5.2**
     * 
     * *For any* form submission attempt, if any required field is empty or null,
     * the validation should return an error for that specific field.
     */
    public function testRequiredFieldValidation(): void
    {
        $requiredFields = [
            'cityId', 'branchId', 'gradeLevel', 'studentFirstName', 
            'studentLastName', 'schoolId', 'parentFirstName', 
            'parentLastName', 'parentPhone1', 'policyAccepted'
        ];

        $this
            ->limitTo(100)
            ->forAll(
                Generator\elements(...$requiredFields)
            )
            ->then(function (string $fieldToOmit) {
                // Create valid data
                $validData = $this->createValidRegistrationData();
                
                // Remove or empty the field to test
                if ($fieldToOmit === 'policyAccepted') {
                    $validData[$fieldToOmit] = false;
                } else {
                    unset($validData[$fieldToOmit]);
                }
                
                $result = $this->validationService->validateRegistration($validData);
                
                $this->assertFalse(
                    $result->isValid(),
                    "Validation should fail when '{$fieldToOmit}' is missing"
                );
                $this->assertTrue(
                    $result->hasError($fieldToOmit),
                    "Should have error for missing field '{$fieldToOmit}'"
                );
            });
    }

    /**
     * **Feature: student-registration-form, Property 7: Conditional Study Field Validation**
     * **Validates: Requirements 2.4**
     * 
     * *For any* form submission where grade level is 10, 11, or 12, 
     * if studyField is null or empty, the validation should return an error.
     * For grade levels 1-9, studyField validation should be skipped.
     */
    public function testConditionalStudyFieldValidation(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                Generator\choose(1, 12) // Grade levels 1-12
            )
            ->then(function (int $gradeLevel) {
                // Create valid data without study field
                $data = $this->createValidRegistrationData();
                $data['gradeLevel'] = $gradeLevel;
                unset($data['studyField']);
                
                $result = $this->validationService->validateRegistration($data);
                
                if ($gradeLevel >= 10 && $gradeLevel <= 12) {
                    // Should require study field for grades 10-12
                    $this->assertTrue(
                        $result->hasError('studyField'),
                        "Grade {$gradeLevel} should require study field"
                    );
                } else {
                    // Should not require study field for grades 1-9
                    $this->assertFalse(
                        $result->hasError('studyField'),
                        "Grade {$gradeLevel} should not require study field"
                    );
                }
            });
    }

    /**
     * **Feature: student-registration-form, Property 9: Optional Phone2 Validation**
     * **Validates: Requirements 4.5**
     * 
     * *For any* form submission, if parentPhone2 is empty or null, validation should pass.
     * If parentPhone2 has a value, it should be validated against the phone format rules.
     */
    public function testOptionalPhone2Validation(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                Generator\oneOf(
                    Generator\constant(null),
                    Generator\constant(''),
                    $this->validPhoneGenerator(),
                    $this->invalidPhoneGenerator()
                )
            )
            ->then(function ($phone2Value) {
                $data = $this->createValidRegistrationData();
                
                if ($phone2Value === null) {
                    unset($data['parentPhone2']);
                } else {
                    $data['parentPhone2'] = $phone2Value;
                }
                
                $result = $this->validationService->validateRegistration($data);
                
                if ($phone2Value === null || $phone2Value === '') {
                    // Empty phone2 should not cause error
                    $this->assertFalse(
                        $result->hasError('parentPhone2'),
                        "Empty phone2 should not cause validation error"
                    );
                } else {
                    // Non-empty phone2 should be validated
                    $isValidPhone = $this->validationService->validatePhoneNumber($phone2Value);
                    $this->assertEquals(
                        !$isValidPhone,
                        $result->hasError('parentPhone2'),
                        "Phone2 validation error should match format validity"
                    );
                }
            });
    }

    /**
     * Generator for valid Turkish phone numbers
     */
    private function validPhoneGenerator(): Generator
    {
        return Generator\map(
            function (array $digits) {
                // First digit is always 5, followed by 9 random digits
                return '5' . implode('', $digits);
            },
            Generator\tuple(
                ...array_fill(0, 9, Generator\choose(0, 9))
            )
        );
    }

    /**
     * Generator for invalid phone numbers
     */
    private function invalidPhoneGenerator(): Generator
    {
        return Generator\oneOf(
            // Too short
            Generator\map(fn($d) => '5' . implode('', $d), 
                Generator\tuple(...array_fill(0, 5, Generator\choose(0, 9)))),
            // Too long
            Generator\map(fn($d) => '5' . implode('', $d), 
                Generator\tuple(...array_fill(0, 12, Generator\choose(0, 9)))),
            // Wrong first digit
            Generator\map(fn($d) => implode('', $d), 
                Generator\tuple(
                    Generator\elements(0, 1, 2, 3, 4, 6, 7, 8, 9),
                    ...array_fill(0, 9, Generator\choose(0, 9))
                ))
        );
    }

    /**
     * Create valid registration data for testing
     */
    private function createValidRegistrationData(): array
    {
        return [
            'cityId' => 1,
            'branchId' => 1,
            'gradeLevel' => 5, // Grade that doesn't require study field
            'studentFirstName' => 'Test',
            'studentLastName' => 'Student',
            'schoolId' => 1,
            'schoolName' => 'Test School',
            'parentFirstName' => 'Test',
            'parentLastName' => 'Parent',
            'parentPhone1' => '5321234567',
            'policyAccepted' => true,
        ];
    }
}
