<?php
/**
 * Property-Based Tests for FormController
 * 
 * **Feature: student-registration-form**
 */

declare(strict_types=1);

namespace Tests\Property;

use PHPUnit\Framework\TestCase;
use Eris\Generator;
use Eris\TestTrait;
use App\Services\DatabaseService;
use App\Services\SchoolApiClient;
use App\Models\Branch;
use App\Models\GradeLevel;
use App\Models\School;
use PDO;

class FormControllerPropertyTest extends TestCase
{
    use TestTrait;

    private PDO $pdo;
    private TestDatabaseServiceForController $databaseService;
    private SchoolApiClient $schoolApiClient;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Use SQLite in-memory database for testing
        $this->pdo = new PDO('sqlite::memory:', null, null, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        
        $this->createTestSchema();
        $this->databaseService = new TestDatabaseServiceForController($this->pdo);
        $this->schoolApiClient = new SchoolApiClient();
    }

    /**
     * Create test database schema
     */
    private function createTestSchema(): void
    {
        // Create cities table
        $this->pdo->exec('
            CREATE TABLE cities (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name VARCHAR(100) NOT NULL,
                code VARCHAR(10) NOT NULL,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ');

        // Create branches table
        $this->pdo->exec('
            CREATE TABLE branches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                city_id INTEGER NOT NULL,
                name VARCHAR(200) NOT NULL,
                address TEXT,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (city_id) REFERENCES cities(id)
            )
        ');

        // Create grade_levels table
        $this->pdo->exec('
            CREATE TABLE grade_levels (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                branch_id INTEGER NOT NULL,
                level INTEGER NOT NULL,
                display_name VARCHAR(50) NOT NULL,
                is_active INTEGER DEFAULT 1,
                FOREIGN KEY (branch_id) REFERENCES branches(id)
            )
        ');
    }

    /**
     * Seed test data with multiple cities, branches, and grade levels
     */
    private function seedTestData(): void
    {
        // Insert cities
        $this->pdo->exec("INSERT INTO cities (id, name, code) VALUES (1, 'İstanbul', '34')");
        $this->pdo->exec("INSERT INTO cities (id, name, code) VALUES (2, 'Ankara', '06')");
        $this->pdo->exec("INSERT INTO cities (id, name, code) VALUES (3, 'İzmir', '35')");
        
        // Insert branches for each city
        // İstanbul branches
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (1, 1, 'Kadıköy Şubesi', 'Kadıköy, İstanbul')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (2, 1, 'Beşiktaş Şubesi', 'Beşiktaş, İstanbul')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (3, 1, 'Üsküdar Şubesi', 'Üsküdar, İstanbul')");
        
        // Ankara branches
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (4, 2, 'Çankaya Şubesi', 'Çankaya, Ankara')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (5, 2, 'Kızılay Şubesi', 'Kızılay, Ankara')");
        
        // İzmir branches
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (6, 3, 'Konak Şubesi', 'Konak, İzmir')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name, address) VALUES (7, 3, 'Karşıyaka Şubesi', 'Karşıyaka, İzmir')");
        
        // Insert grade levels for each branch
        $branchGradeLevels = [
            1 => [1, 2, 3, 4, 5, 9, 10, 11, 12],
            2 => [5, 6, 7, 8, 9, 10],
            3 => [1, 2, 3, 4],
            4 => [9, 10, 11, 12],
            5 => [5, 6, 7, 8],
            6 => [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12],
            7 => [9, 10, 11, 12],
        ];
        
        $gradeLevelId = 1;
        foreach ($branchGradeLevels as $branchId => $levels) {
            foreach ($levels as $level) {
                $displayName = "{$level}. Sınıf";
                $this->pdo->exec("INSERT INTO grade_levels (id, branch_id, level, display_name) VALUES ({$gradeLevelId}, {$branchId}, {$level}, '{$displayName}')");
                $gradeLevelId++;
            }
        }
    }

    /**
     * **Feature: student-registration-form, Property 1: Branch Filtering by City**
     * **Validates: Requirements 1.2**
     * 
     * *For any* city selection, all branches returned by getBranchesByCity(cityId) 
     * should have their cityId field equal to the selected city's ID.
     */
    public function testBranchFilteringByCity(): void
    {
        $this->seedTestData();
        
        $this
            ->limitTo(100)
            ->forAll(
                Generator\choose(1, 3) // Valid city IDs
            )
            ->then(function (int $cityId) {
                $branches = $this->databaseService->getBranchesByCity($cityId);
                
                foreach ($branches as $branch) {
                    $this->assertInstanceOf(Branch::class, $branch);
                    $this->assertEquals(
                        $cityId, 
                        $branch->cityId, 
                        "Branch '{$branch->name}' has cityId {$branch->cityId}, expected {$cityId}"
                    );
                }
            });
    }

    /**
     * **Feature: student-registration-form, Property 3: Grade Level Filtering by Branch**
     * **Validates: Requirements 2.1**
     * 
     * *For any* branch selection, all grade levels returned by getGradeLevelsByBranch(branchId) 
     * should have their branchId field equal to the selected branch's ID.
     */
    public function testGradeLevelFilteringByBranch(): void
    {
        $this->seedTestData();
        
        $this
            ->limitTo(100)
            ->forAll(
                Generator\choose(1, 7) // Valid branch IDs
            )
            ->then(function (int $branchId) {
                $gradeLevels = $this->databaseService->getGradeLevelsByBranch($branchId);
                
                foreach ($gradeLevels as $gradeLevel) {
                    $this->assertInstanceOf(GradeLevel::class, $gradeLevel);
                    $this->assertEquals(
                        $branchId, 
                        $gradeLevel->branchId, 
                        "GradeLevel '{$gradeLevel->displayName}' has branchId {$gradeLevel->branchId}, expected {$branchId}"
                    );
                }
            });
    }

    /**
     * **Feature: student-registration-form, Property 5: School Search Returns City-Filtered Results**
     * **Validates: Requirements 3.2**
     * 
     * *For any* city and search query, all schools returned by searchSchools(cityId, query) 
     * should have their cityId field equal to the selected city's ID and their name 
     * should contain the search query.
     */
    public function testSchoolSearchReturnsCityFilteredResults(): void
    {
        // Set up mock data with schools in different cities
        $this->schoolApiClient->setMockData([
            ['id' => 1, 'name' => 'Atatürk İlkokulu', 'city_id' => 1, 'type' => 'İlkokul'],
            ['id' => 2, 'name' => 'Atatürk Ortaokulu', 'city_id' => 1, 'type' => 'Ortaokul'],
            ['id' => 3, 'name' => 'Atatürk Lisesi', 'city_id' => 2, 'type' => 'Lise'],
            ['id' => 4, 'name' => 'Cumhuriyet İlkokulu', 'city_id' => 1, 'type' => 'İlkokul'],
            ['id' => 5, 'name' => 'Cumhuriyet Ortaokulu', 'city_id' => 2, 'type' => 'Ortaokul'],
            ['id' => 6, 'name' => 'Fatih Anadolu Lisesi', 'city_id' => 1, 'type' => 'Lise'],
            ['id' => 7, 'name' => 'Fatih İlkokulu', 'city_id' => 3, 'type' => 'İlkokul'],
            ['id' => 8, 'name' => 'Konak İlkokulu', 'city_id' => 3, 'type' => 'İlkokul'],
            ['id' => 9, 'name' => 'Karşıyaka Ortaokulu', 'city_id' => 3, 'type' => 'Ortaokul'],
            ['id' => 10, 'name' => 'Çankaya Lisesi', 'city_id' => 2, 'type' => 'Lise'],
        ]);
        
        $this
            ->limitTo(100)
            ->forAll(
                Generator\choose(1, 3), // Valid city IDs
                Generator\elements('ata', 'cum', 'fat', 'kon', 'kar', 'çan', 'il', 'orta', 'lise') // Search queries
            )
            ->then(function (int $cityId, string $query) {
                $schools = $this->schoolApiClient->searchSchools($cityId, $query);
                
                foreach ($schools as $school) {
                    $this->assertInstanceOf(School::class, $school);
                    
                    // Property: cityId must match
                    $this->assertEquals(
                        $cityId, 
                        $school->cityId, 
                        "School '{$school->name}' has cityId {$school->cityId}, expected {$cityId}"
                    );
                    
                    // Property: name must contain query (case-insensitive)
                    $this->assertStringContainsStringIgnoringCase(
                        $query,
                        $school->name,
                        "School name '{$school->name}' should contain query '{$query}'"
                    );
                }
            });
    }

    /**
     * Additional test: Empty results for non-existent city
     */
    public function testBranchFilteringReturnsEmptyForNonExistentCity(): void
    {
        $this->seedTestData();
        
        $this
            ->limitTo(50)
            ->forAll(
                Generator\choose(100, 1000) // Non-existent city IDs
            )
            ->then(function (int $cityId) {
                $branches = $this->databaseService->getBranchesByCity($cityId);
                $this->assertEmpty($branches, "Should return empty array for non-existent city ID {$cityId}");
            });
    }

    /**
     * Additional test: Empty results for non-existent branch
     */
    public function testGradeLevelFilteringReturnsEmptyForNonExistentBranch(): void
    {
        $this->seedTestData();
        
        $this
            ->limitTo(50)
            ->forAll(
                Generator\choose(100, 1000) // Non-existent branch IDs
            )
            ->then(function (int $branchId) {
                $gradeLevels = $this->databaseService->getGradeLevelsByBranch($branchId);
                $this->assertEmpty($gradeLevels, "Should return empty array for non-existent branch ID {$branchId}");
            });
    }
}

/**
 * Test helper class that allows injecting a PDO connection
 */
class TestDatabaseServiceForController extends DatabaseService
{
    private PDO $testPdo;

    public function __construct(PDO $pdo)
    {
        $this->testPdo = $pdo;
        parent::__construct([]);
    }

    public function getConnection(): PDO
    {
        return $this->testPdo;
    }
}
