<?php
/**
 * Property-Based Tests for EncryptionService
 * 
 * **Feature: student-registration-form**
 */

declare(strict_types=1);

namespace Tests\Property;

use PHPUnit\Framework\TestCase;
use Eris\Generator;
use Eris\TestTrait;
use App\Services\EncryptionService;

class EncryptionServicePropertyTest extends TestCase
{
    use TestTrait;

    private EncryptionService $encryptionService;
    private string $testKey;

    protected function setUp(): void
    {
        parent::setUp();
        // Use a fixed 32-character key for testing (exactly 32 chars)
        $this->testKey = 'abcdefghijklmnopqrstuvwxyz123456';
        $this->encryptionService = new EncryptionService($this->testKey);
    }

    /**
     * **Feature: student-registration-form, Property 11: Phone Encryption Round-Trip**
     * **Validates: Requirements 7.4**
     * 
     * *For any* phone number, encrypting and then decrypting should return 
     * the original phone number.
     */
    public function testPhoneEncryptionRoundTrip(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                $this->validPhoneGenerator()
            )
            ->then(function (string $phone) {
                $encrypted = $this->encryptionService->encrypt($phone);
                $decrypted = $this->encryptionService->decrypt($encrypted);
                
                $this->assertEquals(
                    $phone,
                    $decrypted,
                    "Decrypted phone should match original: {$phone}"
                );
            });
    }

    /**
     * **Feature: student-registration-form, Property 11: Phone Encryption Round-Trip (Extended)**
     * **Validates: Requirements 7.4**
     * 
     * *For any* arbitrary string, encrypting and then decrypting should return 
     * the original string. This tests the general round-trip property.
     */
    public function testGeneralEncryptionRoundTrip(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                Generator\string()
            )
            ->then(function (string $plaintext) {
                $encrypted = $this->encryptionService->encrypt($plaintext);
                $decrypted = $this->encryptionService->decrypt($encrypted);
                
                $this->assertEquals(
                    $plaintext,
                    $decrypted,
                    "Decrypted text should match original"
                );
            });
    }

    /**
     * Test that encrypted output is different from input
     */
    public function testEncryptedOutputDiffersFromInput(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                $this->validPhoneGenerator()
            )
            ->then(function (string $phone) {
                $encrypted = $this->encryptionService->encrypt($phone);
                
                // Encrypted output should be different from input
                $this->assertNotEquals(
                    $phone,
                    $encrypted,
                    "Encrypted output should differ from input"
                );
                
                // Encrypted output should be base64 encoded
                $this->assertNotFalse(
                    base64_decode($encrypted, true),
                    "Encrypted output should be valid base64"
                );
            });
    }

    /**
     * Test that same input produces different encrypted outputs (due to random IV)
     */
    public function testEncryptionProducesDifferentOutputs(): void
    {
        $phone = '5321234567';
        
        $encrypted1 = $this->encryptionService->encrypt($phone);
        $encrypted2 = $this->encryptionService->encrypt($phone);
        
        // Due to random IV, same input should produce different encrypted outputs
        $this->assertNotEquals(
            $encrypted1,
            $encrypted2,
            "Same input should produce different encrypted outputs due to random IV"
        );
        
        // But both should decrypt to the same value
        $this->assertEquals($phone, $this->encryptionService->decrypt($encrypted1));
        $this->assertEquals($phone, $this->encryptionService->decrypt($encrypted2));
    }

    /**
     * Generator for valid Turkish phone numbers
     */
    private function validPhoneGenerator(): Generator
    {
        return Generator\map(
            function (array $digits) {
                return '5' . implode('', $digits);
            },
            Generator\tuple(
                ...array_fill(0, 9, Generator\choose(0, 9))
            )
        );
    }
}
