<?php
/**
 * Property-Based Tests for DatabaseService
 * 
 * **Feature: student-registration-form**
 */

declare(strict_types=1);

namespace Tests\Property;

use PHPUnit\Framework\TestCase;
use Eris\Generator;
use Eris\TestTrait;
use App\Services\DatabaseService;
use App\Services\EncryptionService;
use App\Models\Registration;
use App\Models\StudyField;
use PDO;

class DatabaseServicePropertyTest extends TestCase
{
    use TestTrait;

    private DatabaseService $databaseService;
    private EncryptionService $encryptionService;
    private PDO $pdo;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Use SQLite in-memory database for testing
        $this->pdo = new PDO('sqlite::memory:', null, null, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        
        $this->createTestSchema();
        
        // Create encryption service with test key
        $this->encryptionService = new EncryptionService('abcdefghijklmnopqrstuvwxyz123456');
        
        // Create database service with test config
        $config = [
            'driver' => 'sqlite',
            'host' => '',
            'port' => '',
            'database' => ':memory:',
            'username' => '',
            'password' => '',
            'charset' => 'utf8',
            'options' => [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            ],
        ];
        
        $this->databaseService = new TestDatabaseService($this->pdo, $this->encryptionService);
    }


    /**
     * Create test database schema
     */
    private function createTestSchema(): void
    {
        // Create cities table
        $this->pdo->exec('
            CREATE TABLE cities (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name VARCHAR(100) NOT NULL,
                code VARCHAR(10) NOT NULL,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ');

        // Create branches table
        $this->pdo->exec('
            CREATE TABLE branches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                city_id INTEGER NOT NULL,
                name VARCHAR(200) NOT NULL,
                address TEXT,
                is_active INTEGER DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (city_id) REFERENCES cities(id)
            )
        ');

        // Create grade_levels table
        $this->pdo->exec('
            CREATE TABLE grade_levels (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                branch_id INTEGER NOT NULL,
                level INTEGER NOT NULL,
                display_name VARCHAR(50) NOT NULL,
                is_active INTEGER DEFAULT 1,
                FOREIGN KEY (branch_id) REFERENCES branches(id)
            )
        ');

        // Create registrations table
        $this->pdo->exec('
            CREATE TABLE registrations (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                city_id INTEGER NOT NULL,
                branch_id INTEGER NOT NULL,
                grade_level INTEGER NOT NULL,
                study_field VARCHAR(20),
                student_first_name VARCHAR(100) NOT NULL,
                student_last_name VARCHAR(100) NOT NULL,
                school_id INTEGER NOT NULL,
                school_name VARCHAR(200) NOT NULL,
                parent_first_name VARCHAR(100) NOT NULL,
                parent_last_name VARCHAR(100) NOT NULL,
                parent_phone1_encrypted TEXT NOT NULL,
                parent_phone2_encrypted TEXT,
                policy_accepted INTEGER NOT NULL DEFAULT 1,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (city_id) REFERENCES cities(id),
                FOREIGN KEY (branch_id) REFERENCES branches(id)
            )
        ');

        // Insert test data
        $this->pdo->exec("INSERT INTO cities (id, name, code) VALUES (1, 'İstanbul', '34')");
        $this->pdo->exec("INSERT INTO cities (id, name, code) VALUES (2, 'Ankara', '06')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name) VALUES (1, 1, 'Kadıköy Şubesi')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name) VALUES (2, 1, 'Beşiktaş Şubesi')");
        $this->pdo->exec("INSERT INTO branches (id, city_id, name) VALUES (3, 2, 'Çankaya Şubesi')");
        $this->pdo->exec("INSERT INTO grade_levels (id, branch_id, level, display_name) VALUES (1, 1, 9, '9. Sınıf')");
        $this->pdo->exec("INSERT INTO grade_levels (id, branch_id, level, display_name) VALUES (2, 1, 10, '10. Sınıf')");
        $this->pdo->exec("INSERT INTO grade_levels (id, branch_id, level, display_name) VALUES (3, 2, 11, '11. Sınıf')");
    }


    /**
     * **Feature: student-registration-form, Property 10: Registration Data Round-Trip**
     * **Validates: Requirements 7.1**
     * 
     * *For any* valid registration data, after saving to database and retrieving,
     * the retrieved data should be equivalent to the original data 
     * (excluding auto-generated fields like id and createdAt).
     */
    public function testRegistrationDataRoundTrip(): void
    {
        $this
            ->limitTo(100)
            ->forAll(
                $this->registrationGenerator()
            )
            ->then(function (Registration $registration) {
                // Save registration
                $id = $this->databaseService->saveRegistration($registration);
                
                // Retrieve registration
                $retrieved = $this->databaseService->getRegistrationById($id);
                
                $this->assertNotNull($retrieved, "Retrieved registration should not be null");
                
                // Compare fields (excluding auto-generated id and createdAt)
                $this->assertEquals($registration->cityId, $retrieved->cityId, "cityId mismatch");
                $this->assertEquals($registration->branchId, $retrieved->branchId, "branchId mismatch");
                $this->assertEquals($registration->gradeLevel, $retrieved->gradeLevel, "gradeLevel mismatch");
                $this->assertEquals($registration->studyField, $retrieved->studyField, "studyField mismatch");
                $this->assertEquals($registration->studentFirstName, $retrieved->studentFirstName, "studentFirstName mismatch");
                $this->assertEquals($registration->studentLastName, $retrieved->studentLastName, "studentLastName mismatch");
                $this->assertEquals($registration->schoolId, $retrieved->schoolId, "schoolId mismatch");
                $this->assertEquals($registration->schoolName, $retrieved->schoolName, "schoolName mismatch");
                $this->assertEquals($registration->parentFirstName, $retrieved->parentFirstName, "parentFirstName mismatch");
                $this->assertEquals($registration->parentLastName, $retrieved->parentLastName, "parentLastName mismatch");
                $this->assertEquals($registration->parentPhone1, $retrieved->parentPhone1, "parentPhone1 mismatch");
                $this->assertEquals($registration->parentPhone2, $retrieved->parentPhone2, "parentPhone2 mismatch");
                $this->assertEquals($registration->policyAccepted, $retrieved->policyAccepted, "policyAccepted mismatch");
            });
    }

    /**
     * Generator for valid Registration objects
     */
    private function registrationGenerator(): Generator
    {
        return Generator\bind(
            Generator\tuple(
                Generator\choose(1, 2),           // cityId
                Generator\choose(1, 3),           // branchId
                Generator\choose(1, 12),          // gradeLevel
                $this->nameGenerator(),           // studentFirstName
                $this->nameGenerator(),           // studentLastName
                Generator\choose(1, 1000),        // schoolId
                $this->schoolNameGenerator(),     // schoolName
                $this->nameGenerator(),           // parentFirstName
                $this->nameGenerator(),           // parentLastName
                $this->validPhoneGenerator(),     // parentPhone1
                Generator\oneOf(                  // parentPhone2 (optional)
                    Generator\constant(null),
                    $this->validPhoneGenerator()
                )
            ),
            function (array $data) {
                [$cityId, $branchId, $gradeLevel, $studentFirst, $studentLast, 
                 $schoolId, $schoolName, $parentFirst, $parentLast, $phone1, $phone2] = $data;
                
                // Determine study field based on grade level
                $studyField = null;
                if ($gradeLevel >= 10 && $gradeLevel <= 12) {
                    $fields = StudyField::getAll();
                    $studyField = $fields[array_rand($fields)];
                }
                
                return Generator\constant(new Registration(
                    id: null,
                    cityId: $cityId,
                    branchId: $branchId,
                    gradeLevel: $gradeLevel,
                    studyField: $studyField,
                    studentFirstName: $studentFirst,
                    studentLastName: $studentLast,
                    schoolId: $schoolId,
                    schoolName: $schoolName,
                    parentFirstName: $parentFirst,
                    parentLastName: $parentLast,
                    parentPhone1: $phone1,
                    parentPhone2: $phone2,
                    policyAccepted: true,
                    createdAt: null
                ));
            }
        );
    }

    /**
     * Generator for valid Turkish phone numbers
     */
    private function validPhoneGenerator(): Generator
    {
        return Generator\map(
            function (array $digits) {
                return '5' . implode('', $digits);
            },
            Generator\tuple(
                ...array_fill(0, 9, Generator\choose(0, 9))
            )
        );
    }

    /**
     * Generator for names (Turkish-safe characters)
     */
    private function nameGenerator(): Generator
    {
        return Generator\map(
            function (int $length) {
                $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
                $name = '';
                for ($i = 0; $i < $length; $i++) {
                    $name .= $chars[random_int(0, strlen($chars) - 1)];
                }
                return ucfirst(strtolower($name));
            },
            Generator\choose(2, 20)
        );
    }

    /**
     * Generator for school names
     */
    private function schoolNameGenerator(): Generator
    {
        return Generator\map(
            function (array $parts) {
                return implode(' ', $parts) . ' Okulu';
            },
            Generator\tuple(
                $this->nameGenerator(),
                Generator\elements('İlkokulu', 'Ortaokulu', 'Lisesi', 'Koleji')
            )
        );
    }
}


/**
 * Test helper class that allows injecting a PDO connection
 */
class TestDatabaseService extends DatabaseService
{
    private PDO $testPdo;
    private ?EncryptionService $testEncryptionService;

    public function __construct(PDO $pdo, ?EncryptionService $encryptionService = null)
    {
        $this->testPdo = $pdo;
        $this->testEncryptionService = $encryptionService;
        parent::__construct([], $encryptionService);
    }

    public function getConnection(): PDO
    {
        return $this->testPdo;
    }
}
