<?php
/**
 * SchoolApiClient
 * 
 * MEB Okul API entegrasyonu
 * Requirements: 3.2
 * Property 5: School Search Returns City-Filtered Results
 */

declare(strict_types=1);

namespace App\Services;

use App\Models\School;

class SchoolApiClient
{
    private const MEB_API_URL = 'https://www.meb.gov.tr/baglantilar/okullar/okullar_ajax.php';
    
    /**
     * Şehir ID -> Plaka kodu eşleştirmesi
     * Veritabanındaki city.id ile plaka kodunu eşleştiriyoruz
     */
    private array $cityPlateMap = [];
    
    private bool $useMockData = false;
    private array $mockSchools = [];

    public function __construct(?array $cityPlateMap = null)
    {
        // Şehir-plaka eşleştirmesi veritabanından veya parametre olarak alınır
        // Veritabanındaki cities.code alanı plaka kodunu içerir
        $this->cityPlateMap = $cityPlateMap ?? [];
    }

    /**
     * Şehir ID'sinden plaka kodunu al (veritabanı code alanından)
     */
    public function setPlateCodeFromDatabase(int $cityId, string $plateCode): void
    {
        $this->cityPlateMap[$cityId] = (int) $plateCode;
    }

    /**
     * Search schools by city and query using MEB API
     * 
     * Requirements: 3.2
     * Property 5: All returned schools must have cityId equal to the parameter
     *             and name containing the search query
     * 
     * @param int $cityId City ID to filter by
     * @param string $query Search query for school name
     * @return School[]
     */
    public function searchSchools(int $cityId, string $query): array
    {
        error_log("[SchoolApiClient] searchSchools called - cityId: $cityId, query: $query");
        
        // Test modunda mock data kullan
        if ($this->useMockData) {
            error_log("[SchoolApiClient] Using mock data");
            return $this->searchMockSchools($cityId, $query);
        }

        $query = trim($query);
        
        if (empty($query) || mb_strlen($query) < 2) {
            error_log("[SchoolApiClient] Query too short, returning empty");
            return [];
        }

        // Şehir ID'sinden plaka kodunu al
        $plateCode = $this->cityPlateMap[$cityId] ?? null;
        error_log("[SchoolApiClient] Plate code for cityId $cityId: " . ($plateCode ?? 'NULL'));
        error_log("[SchoolApiClient] cityPlateMap: " . json_encode($this->cityPlateMap));
        
        if ($plateCode === null) {
            error_log("[SchoolApiClient] No plate code found, returning empty");
            return [];
        }

        try {
            error_log("[SchoolApiClient] Fetching from MEB API...");
            $results = $this->fetchFromMebApi($plateCode, $query, $cityId);
            error_log("[SchoolApiClient] Got " . count($results) . " results from MEB API");
            return $results;
        } catch (\Exception $e) {
            // API hatası durumunda boş dizi dön
            error_log('[SchoolApiClient] MEB API Error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * MEB API'den okul verilerini çek
     */
    private function fetchFromMebApi(int $plateCode, string $query, int $cityId): array
    {
        $postData = $this->buildPostData($plateCode, $query);
        error_log("[SchoolApiClient] POST data: " . $postData);
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => self::MEB_API_URL,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postData,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
                'Accept: application/json, text/javascript, */*; q=0.01',
                'Accept-Language: tr-TR,tr;q=0.9,en-US;q=0.8,en;q=0.7',
                'Origin: https://www.meb.gov.tr',
                'Referer: https://www.meb.gov.tr/baglantilar/okullar/',
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'X-Requested-With: XMLHttpRequest',
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true,
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        error_log("[SchoolApiClient] HTTP Code: $httpCode");
        
        if ($error) {
            error_log("[SchoolApiClient] cURL Error: $error");
            throw new \RuntimeException('cURL Error: ' . $error);
        }
        
        if ($httpCode !== 200) {
            error_log("[SchoolApiClient] HTTP Error: $httpCode, Response: " . substr($response, 0, 500));
            throw new \RuntimeException('HTTP Error: ' . $httpCode);
        }
        
        error_log("[SchoolApiClient] Response (first 500 chars): " . substr($response, 0, 500));
        
        return $this->parseResponse($response, $cityId);
    }

    /**
     * MEB API için POST verisi oluştur
     */
    private function buildPostData(int $plateCode, string $query): string
    {
        $data = [
            'draw' => 1,
            'start' => 0,
            'length' => 25,
            'search' => ['value' => $query, 'regex' => 'false'],
            'il' => $plateCode,
            'ilce' => 0,
        ];
        
        // DataTables formatında columns ekle
        for ($i = 0; $i < 3; $i++) {
            $data["columns[$i][data]"] = 'OKUL_ADI';
            $data["columns[$i][name]"] = '';
            $data["columns[$i][searchable]"] = 'true';
            $data["columns[$i][orderable]"] = 'true';
            $data["columns[$i][search][value]"] = '';
            $data["columns[$i][search][regex]"] = 'false';
        }
        
        $data['order[0][column]'] = 0;
        $data['order[0][dir]'] = 'asc';
        $data['order[0][name]'] = '';
        
        return http_build_query($data);
    }

    /**
     * MEB API yanıtını parse et
     */
    private function parseResponse(string $response, int $cityId): array
    {
        $json = json_decode($response, true);
        
        if (!$json || !isset($json['data'])) {
            return [];
        }
        
        $schools = [];
        
        foreach ($json['data'] as $index => $item) {
            // MEB API'den gelen veri yapısı: "ANKARA - AKYURT - Akyurt Anadolu Lisesi"
            $fullName = $item['OKUL_ADI'] ?? null;
            $host = $item['HOST'] ?? null;
            
            if (!$fullName) {
                continue;
            }
            
            // HTML etiketlerini temizle
            $fullName = strip_tags($fullName);
            $fullName = html_entity_decode($fullName, ENT_QUOTES, 'UTF-8');
            $fullName = trim($fullName);
            
            // "ANKARA - AKYURT - Okul Adı" formatını parse et
            $parts = explode(' - ', $fullName);
            
            // Son kısım okul adı
            $schoolName = count($parts) >= 3 ? trim(end($parts)) : $fullName;
            
            // İlçe bilgisi (varsa)
            $district = count($parts) >= 2 ? trim($parts[1]) : '';
            
            // Okul türünü belirle
            $type = $this->determineSchoolType($schoolName);
            
            // Benzersiz ID oluştur (host'tan veya index'ten)
            // crc32() INT max'ı aşabilir, modulo ile sınırla
            $uniqueId = $host ? abs(crc32($host)) % 2000000000 : ($index + 1);
            
            $schools[] = new School(
                id: $uniqueId,
                name: $schoolName,
                cityId: $cityId,
                type: $type,
                district: $district
            );
        }
        
        return $schools;
    }

    /**
     * Okul adından türünü belirle
     */
    private function determineSchoolType(string $name): string
    {
        $nameLower = mb_strtolower($name, 'UTF-8');
        
        if (str_contains($nameLower, 'ilkokul') || str_contains($nameLower, 'ilköğretim')) {
            return 'İlkokul';
        }
        if (str_contains($nameLower, 'ortaokul')) {
            return 'Ortaokul';
        }
        if (str_contains($nameLower, 'lise') || str_contains($nameLower, 'lisesi')) {
            return 'Lise';
        }
        if (str_contains($nameLower, 'anaokul')) {
            return 'Anaokulu';
        }
        
        return 'Okul';
    }

    /**
     * Get school by ID
     * Not: MEB API'de ID ile arama yok, bu metod sadece form submit için kullanılıyor
     * 
     * @param int $schoolId
     * @return School|null
     */
    public function getSchoolById(int $schoolId): ?School
    {
        // MEB API'de ID ile arama yapılamıyor
        // Form submit sırasında schoolName zaten gönderiliyor
        return null;
    }

    /**
     * Şehir-plaka eşleştirmesini güncelle
     */
    public function setCityPlateMap(array $map): void
    {
        $this->cityPlateMap = $map;
    }

    /**
     * Plaka kodunu al
     */
    public function getPlateCode(int $cityId): ?int
    {
        return $this->cityPlateMap[$cityId] ?? null;
    }

    // ============ Test metodları ============

    /**
     * Mock modu aktif et (testler için)
     */
    public function enableMockMode(): void
    {
        $this->useMockData = true;
    }

    /**
     * Mock modu deaktif et
     */
    public function disableMockMode(): void
    {
        $this->useMockData = false;
    }

    /**
     * Mock data ile arama (testler için)
     */
    private function searchMockSchools(int $cityId, string $query): array
    {
        $query = mb_strtolower(trim($query), 'UTF-8');
        
        if (empty($query) || strlen($query) < 2) {
            return [];
        }

        $results = [];
        
        foreach ($this->mockSchools as $school) {
            if ($school['city_id'] !== $cityId) {
                continue;
            }
            
            $schoolName = mb_strtolower($school['name'], 'UTF-8');
            if (mb_strpos($schoolName, $query) !== false) {
                $results[] = School::fromArray($school);
            }
        }

        return $results;
    }

    /**
     * Mock data ekle (testler için)
     */
    public function addMockSchool(array $schoolData): void
    {
        $this->mockSchools[] = $schoolData;
    }

    /**
     * Mock data temizle (testler için)
     */
    public function clearMockData(): void
    {
        $this->mockSchools = [];
    }

    /**
     * Mock data ayarla (testler için)
     */
    public function setMockData(array $schools): void
    {
        $this->mockSchools = $schools;
    }
}
