<?php
/**
 * EncryptionService
 * 
 * Requirements: 7.4
 * Property 11: Phone Encryption Round-Trip
 */

declare(strict_types=1);

namespace App\Services;

use RuntimeException;

class EncryptionService
{
    /**
     * Encryption algorithm (AES-256-CBC)
     */
    private const CIPHER = 'aes-256-cbc';

    /**
     * IV length for AES-256-CBC
     */
    private const IV_LENGTH = 16;

    /**
     * Encryption key (must be 32 bytes for AES-256)
     */
    private string $key;

    /**
     * @param string $key Encryption key (32 characters for AES-256)
     * @throws RuntimeException If key is invalid
     */
    public function __construct(string $key)
    {
        if (strlen($key) !== 32) {
            throw new RuntimeException(
                'Encryption key must be exactly 32 characters for AES-256'
            );
        }
        $this->key = $key;
    }

    /**
     * Create EncryptionService from application config
     * 
     * @return self
     * @throws RuntimeException If encryption key is not configured
     */
    public static function fromConfig(): self
    {
        $config = require __DIR__ . '/../../config/app.php';
        $key = $config['encryption_key'] ?? '';
        
        if (empty($key)) {
            throw new RuntimeException(
                'Encryption key not configured. Set ENCRYPTION_KEY in .env file.'
            );
        }
        
        return new self($key);
    }

    /**
     * Encrypt a string using AES-256-CBC
     * 
     * Requirements: 7.4
     * 
     * @param string $plaintext The string to encrypt
     * @return string Base64-encoded encrypted string (IV + ciphertext)
     * @throws RuntimeException If encryption fails
     */
    public function encrypt(string $plaintext): string
    {
        // Generate a random IV
        $iv = openssl_random_pseudo_bytes(self::IV_LENGTH);
        
        if ($iv === false) {
            throw new RuntimeException('Failed to generate IV');
        }

        // Encrypt the plaintext
        $ciphertext = openssl_encrypt(
            $plaintext,
            self::CIPHER,
            $this->key,
            OPENSSL_RAW_DATA,
            $iv
        );

        if ($ciphertext === false) {
            throw new RuntimeException('Encryption failed');
        }

        // Prepend IV to ciphertext and base64 encode
        return base64_encode($iv . $ciphertext);
    }

    /**
     * Decrypt a string encrypted with encrypt()
     * 
     * Requirements: 7.4
     * Property 11: Phone Encryption Round-Trip - decrypt(encrypt(x)) === x
     * 
     * @param string $encrypted Base64-encoded encrypted string
     * @return string The decrypted plaintext
     * @throws RuntimeException If decryption fails
     */
    public function decrypt(string $encrypted): string
    {
        // Decode from base64
        $data = base64_decode($encrypted, true);
        
        if ($data === false) {
            throw new RuntimeException('Invalid base64 encoding');
        }

        // Extract IV and ciphertext
        if (strlen($data) < self::IV_LENGTH) {
            throw new RuntimeException('Invalid encrypted data: too short');
        }

        $iv = substr($data, 0, self::IV_LENGTH);
        $ciphertext = substr($data, self::IV_LENGTH);

        // Decrypt
        $plaintext = openssl_decrypt(
            $ciphertext,
            self::CIPHER,
            $this->key,
            OPENSSL_RAW_DATA,
            $iv
        );

        if ($plaintext === false) {
            throw new RuntimeException('Decryption failed');
        }

        return $plaintext;
    }

    /**
     * Generate a random encryption key
     * 
     * @return string 32-character key suitable for AES-256
     */
    public static function generateKey(): string
    {
        return bin2hex(openssl_random_pseudo_bytes(16));
    }
}
