<?php
/**
 * DatabaseService
 * 
 * Veritabanı işlemleri
 */

declare(strict_types=1);

namespace App\Services;

use App\Models\City;
use App\Models\Branch;
use App\Models\GradeLevel;
use App\Models\ExamType;
use App\Models\ExamSession;
use App\Models\Registration;
use PDO;
use PDOException;

class DatabaseService
{
    private ?PDO $pdo = null;
    private array $config;
    private ?EncryptionService $encryptionService = null;

    public function __construct(?array $config = null, ?EncryptionService $encryptionService = null)
    {
        $this->config = $config ?? require __DIR__ . '/../../config/database.php';
        $this->encryptionService = $encryptionService;
    }

    /**
     * Get PDO connection (lazy initialization)
     */
    public function getConnection(): PDO
    {
        if ($this->pdo === null) {
            $dsn = sprintf(
                '%s:host=%s;port=%s;dbname=%s;charset=%s',
                $this->config['driver'],
                $this->config['host'],
                $this->config['port'],
                $this->config['database'],
                $this->config['charset']
            );

            $this->pdo = new PDO(
                $dsn,
                $this->config['username'],
                $this->config['password'],
                $this->config['options']
            );
        }

        return $this->pdo;
    }

    /**
     * Set encryption service
     */
    public function setEncryptionService(EncryptionService $encryptionService): void
    {
        $this->encryptionService = $encryptionService;
    }

    /**
     * Get all active cities that have active exam sessions
     * 
     * @return City[]
     */
    public function getCities(): array
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT DISTINCT c.* 
             FROM cities c
             INNER JOIN exam_sessions es ON c.id = es.city_id
             WHERE c.is_active = 1 
             AND es.is_active = 1
             AND es.exam_date >= CURDATE()
             ORDER BY c.name ASC'
        );
        $stmt->execute();
        
        $cities = [];
        while ($row = $stmt->fetch()) {
            $cities[] = City::fromArray($row);
        }
        
        return $cities;
    }

    // ============ Exam Type Methods ============

    /**
     * Get all active exam types with their grade levels
     * 
     * @return ExamType[]
     */
    public function getExamTypes(): array
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM exam_types WHERE is_active = 1 ORDER BY name ASC'
        );
        $stmt->execute();
        
        $examTypes = [];
        while ($row = $stmt->fetch()) {
            $examType = ExamType::fromArray($row);
            // Load grade levels for this exam type
            $examType->gradeLevels = $this->getGradeLevelsByExamType($examType->id);
            $examTypes[] = $examType;
        }
        
        return $examTypes;
    }

    /**
     * Get exam type by ID
     */
    public function getExamTypeById(int $id): ?ExamType
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM exam_types WHERE id = :id AND is_active = 1'
        );
        $stmt->execute(['id' => $id]);
        
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        $examType = ExamType::fromArray($row);
        $examType->gradeLevels = $this->getGradeLevelsByExamType($examType->id);
        
        return $examType;
    }

    // ============ Exam Session Methods ============

    /**
     * Get exam sessions by city and optionally exam type
     * 
     * @param int $cityId
     * @param int|null $examTypeId
     * @return ExamSession[]
     */
    public function getExamSessions(int $cityId, ?int $examTypeId = null): array
    {
        $sql = 'SELECT es.*, et.name as exam_type_name, c.name as city_name,
                       COUNT(r.id) as registration_count
                FROM exam_sessions es
                JOIN exam_types et ON es.exam_type_id = et.id
                JOIN cities c ON es.city_id = c.id
                LEFT JOIN registrations r ON es.id = r.exam_session_id
                WHERE es.city_id = :city_id 
                AND es.is_active = 1 
                AND es.exam_date >= CURDATE()';
        
        $params = ['city_id' => $cityId];
        
        if ($examTypeId !== null) {
            $sql .= ' AND es.exam_type_id = :exam_type_id';
            $params['exam_type_id'] = $examTypeId;
        }
        
        $sql .= ' GROUP BY es.id
                  HAVING (es.capacity IS NULL OR registration_count < es.capacity)
                  ORDER BY es.exam_date ASC, es.exam_time ASC';
        
        $stmt = $this->getConnection()->prepare($sql);
        $stmt->execute($params);
        
        $sessions = [];
        while ($row = $stmt->fetch()) {
            $sessions[] = ExamSession::fromArray($row);
        }
        
        return $sessions;
    }

    /**
     * Get exam session by ID
     */
    public function getExamSessionById(int $id): ?ExamSession
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT es.*, et.name as exam_type_name, c.name as city_name
             FROM exam_sessions es
             JOIN exam_types et ON es.exam_type_id = et.id
             JOIN cities c ON es.city_id = c.id
             WHERE es.id = :id'
        );
        $stmt->execute(['id' => $id]);
        
        $row = $stmt->fetch();
        return $row ? ExamSession::fromArray($row) : null;
    }

    /**
     * Increment exam session registered count
     */
    public function incrementExamSessionCount(int $sessionId): void
    {
        $stmt = $this->getConnection()->prepare(
            'UPDATE exam_sessions SET registered_count = registered_count + 1 WHERE id = :id'
        );
        $stmt->execute(['id' => $sessionId]);
    }

    // ============ Grade Level Methods ============

    /**
     * Get grade levels by exam type ID
     * 
     * @param int $examTypeId
     * @return GradeLevel[]
     */
    public function getGradeLevelsByExamType(int $examTypeId): array
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM exam_type_grade_levels 
             WHERE exam_type_id = :exam_type_id 
             ORDER BY grade_level ASC'
        );
        $stmt->execute(['exam_type_id' => $examTypeId]);
        
        $gradeLevels = [];
        while ($row = $stmt->fetch()) {
            $gradeLevels[] = GradeLevel::fromArray($row);
        }
        
        return $gradeLevels;
    }

    // ============ Legacy Methods (backward compatibility) ============

    /**
     * Get branches by city ID
     * @deprecated Use getExamSessions instead
     */
    public function getBranchesByCity(int $cityId): array
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM branches WHERE city_id = :city_id AND is_active = 1 ORDER BY name ASC'
        );
        $stmt->execute(['city_id' => $cityId]);
        
        $branches = [];
        while ($row = $stmt->fetch()) {
            $branches[] = Branch::fromArray($row);
        }
        
        return $branches;
    }

    /**
     * Get grade levels by branch ID
     * @deprecated Use getGradeLevelsByExamType instead
     */
    public function getGradeLevelsByBranch(int $branchId): array
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM grade_levels WHERE branch_id = :branch_id AND is_active = 1 ORDER BY level ASC'
        );
        $stmt->execute(['branch_id' => $branchId]);
        
        $gradeLevels = [];
        while ($row = $stmt->fetch()) {
            $gradeLevels[] = GradeLevel::fromArray($row);
        }
        
        return $gradeLevels;
    }

    // ============ Registration Methods ============

    /**
     * Save registration to database
     */
    public function saveRegistration(Registration $registration): int
    {
        // Encrypt phone numbers if encryption service is available
        $phone1Encrypted = $this->encryptPhone($registration->parentPhone1);
        $phone2Encrypted = $registration->parentPhone2 
            ? $this->encryptPhone($registration->parentPhone2) 
            : null;

        $sql = 'INSERT INTO registrations (
            city_id, exam_session_id, exam_type_id, grade_level, study_field,
            student_first_name, student_last_name, school_id, school_name,
            parent_first_name, parent_last_name,
            parent_phone1_encrypted, parent_phone2_encrypted,
            policy_accepted, created_at
        ) VALUES (
            :city_id, :exam_session_id, :exam_type_id, :grade_level, :study_field,
            :student_first_name, :student_last_name, :school_id, :school_name,
            :parent_first_name, :parent_last_name,
            :parent_phone1_encrypted, :parent_phone2_encrypted,
            :policy_accepted, :created_at
        )';

        $stmt = $this->getConnection()->prepare($sql);
        $stmt->execute([
            'city_id' => $registration->cityId,
            'exam_session_id' => $registration->examSessionId,
            'exam_type_id' => $registration->examTypeId,
            'grade_level' => $registration->gradeLevel,
            'study_field' => $registration->studyField,
            'student_first_name' => $registration->studentFirstName,
            'student_last_name' => $registration->studentLastName,
            'school_id' => $registration->schoolId,
            'school_name' => $registration->schoolName,
            'parent_first_name' => $registration->parentFirstName,
            'parent_last_name' => $registration->parentLastName,
            'parent_phone1_encrypted' => $phone1Encrypted,
            'parent_phone2_encrypted' => $phone2Encrypted,
            'policy_accepted' => $registration->policyAccepted ? 1 : 0,
            'created_at' => date('Y-m-d H:i:s'),
        ]);

        return (int) $this->getConnection()->lastInsertId();
    }

    /**
     * Get registration by ID
     */
    public function getRegistrationById(int $id): ?Registration
    {
        $stmt = $this->getConnection()->prepare(
            'SELECT * FROM registrations WHERE id = :id'
        );
        $stmt->execute(['id' => $id]);
        
        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        // Decrypt phone numbers
        $row['parent_phone1'] = $this->decryptPhone($row['parent_phone1_encrypted']);
        $row['parent_phone2'] = $row['parent_phone2_encrypted'] 
            ? $this->decryptPhone($row['parent_phone2_encrypted']) 
            : null;

        return Registration::fromArray($row);
    }

    /**
     * Encrypt phone number
     */
    private function encryptPhone(string $phone): string
    {
        if ($this->encryptionService === null) {
            return $phone;
        }
        return $this->encryptionService->encrypt($phone);
    }

    /**
     * Decrypt phone number
     */
    private function decryptPhone(string $encryptedPhone): string
    {
        if ($this->encryptionService === null) {
            return $encryptedPhone;
        }
        return $this->encryptionService->decrypt($encryptedPhone);
    }

    /**
     * Close the database connection
     */
    public function close(): void
    {
        $this->pdo = null;
    }
}
