<?php
/**
 * Simple PHP Router
 */

declare(strict_types=1);

namespace App;

class Router
{
    private array $routes = [];

    public function get(string $path, callable|array $handler): void
    {
        $this->routes['GET'][$path] = $handler;
    }

    public function post(string $path, callable|array $handler): void
    {
        $this->routes['POST'][$path] = $handler;
    }

    public function put(string $path, callable|array $handler): void
    {
        $this->routes['PUT'][$path] = $handler;
    }

    public function delete(string $path, callable|array $handler): void
    {
        $this->routes['DELETE'][$path] = $handler;
    }

    public function dispatch(string $method, string $uri): void
    {
        $uri = rtrim($uri, '/') ?: '/';
        
        // Debug: log the request
        $debugMsg = "Router: $method $uri\n";
        $debugMsg .= "Available routes for $method: " . implode(', ', array_keys($this->routes[$method] ?? [])) . "\n";
        file_put_contents('/tmp/router-debug.txt', date('Y-m-d H:i:s') . " - " . $debugMsg, FILE_APPEND);
        
        // Check for exact match first
        if (isset($this->routes[$method][$uri])) {
            file_put_contents('/tmp/router-debug.txt', "Exact match found for $uri\n", FILE_APPEND);
            $this->callHandler($this->routes[$method][$uri]);
            return;
        }

        // Check for parameterized routes
        foreach ($this->routes[$method] ?? [] as $route => $handler) {
            $pattern = $this->convertRouteToRegex($route);
            file_put_contents('/tmp/router-debug.txt', "Testing route: $route with pattern: $pattern\n", FILE_APPEND);
            if (preg_match($pattern, $uri, $matches)) {
                file_put_contents('/tmp/router-debug.txt', "Pattern match found for $uri with route $route\n", FILE_APPEND);
                array_shift($matches); // Remove full match
                $this->callHandler($handler, $matches);
                return;
            }
        }

        // 404 Not Found
        file_put_contents('/tmp/router-debug.txt', "No match found for $method $uri\n", FILE_APPEND);
        http_response_code(404);
        echo json_encode(['error' => 'Not Found']);
    }

    private function convertRouteToRegex(string $route): string
    {
        // Escape forward slashes and convert {param} to capture groups
        $pattern = preg_replace('/\{(\w+)\}/', '([^/]+)', $route);
        return '#^' . $pattern . '$#';
    }

    private function callHandler(callable|array $handler, array $params = []): void
    {
        if (is_array($handler)) {
            [$class, $method] = $handler;
            $instance = new $class();
            call_user_func_array([$instance, $method], $params);
        } else {
            call_user_func_array($handler, $params);
        }
    }
}
