<?php
/**
 * Registration Model
 * 
 * Sınav kayıt modeli
 */

declare(strict_types=1);

namespace App\Models;

use DateTime;

class Registration
{
    public function __construct(
        public ?int $id = null,
        public int $cityId = 0,
        public int $examSessionId = 0,
        public int $examTypeId = 0,
        public int $gradeLevel = 0,
        public ?string $studyField = null,
        public string $studentFirstName = '',
        public string $studentLastName = '',
        public int $schoolId = 0,
        public string $schoolName = '',
        public string $parentFirstName = '',
        public string $parentLastName = '',
        public string $parentPhone1 = '',
        public ?string $parentPhone2 = null,
        public bool $policyAccepted = false,
        public ?DateTime $createdAt = null
    ) {}

    /**
     * Create Registration from database row
     */
    public static function fromArray(array $data): self
    {
        return new self(
            id: isset($data['id']) ? (int) $data['id'] : null,
            cityId: isset($data['city_id']) ? (int) $data['city_id'] : 0,
            examSessionId: isset($data['exam_session_id']) ? (int) $data['exam_session_id'] : 0,
            examTypeId: isset($data['exam_type_id']) ? (int) $data['exam_type_id'] : 0,
            gradeLevel: isset($data['grade_level']) ? (int) $data['grade_level'] : 0,
            studyField: $data['study_field'] ?? null,
            studentFirstName: $data['student_first_name'] ?? '',
            studentLastName: $data['student_last_name'] ?? '',
            schoolId: isset($data['school_id']) ? (int) $data['school_id'] : 0,
            schoolName: $data['school_name'] ?? '',
            parentFirstName: $data['parent_first_name'] ?? '',
            parentLastName: $data['parent_last_name'] ?? '',
            parentPhone1: $data['parent_phone1'] ?? '',
            parentPhone2: $data['parent_phone2'] ?? null,
            policyAccepted: isset($data['policy_accepted']) ? (bool) $data['policy_accepted'] : false,
            createdAt: isset($data['created_at']) ? new DateTime($data['created_at']) : null
        );
    }

    /**
     * Convert to array for database operations
     */
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'city_id' => $this->cityId,
            'exam_session_id' => $this->examSessionId,
            'exam_type_id' => $this->examTypeId,
            'grade_level' => $this->gradeLevel,
            'study_field' => $this->studyField,
            'student_first_name' => $this->studentFirstName,
            'student_last_name' => $this->studentLastName,
            'school_id' => $this->schoolId,
            'school_name' => $this->schoolName,
            'parent_first_name' => $this->parentFirstName,
            'parent_last_name' => $this->parentLastName,
            'parent_phone1' => $this->parentPhone1,
            'parent_phone2' => $this->parentPhone2,
            'policy_accepted' => $this->policyAccepted,
            'created_at' => $this->createdAt?->format('Y-m-d H:i:s'),
        ];
    }

    /**
     * Check if study field is required based on grade level
     * 10-12. sınıflar için alan seçimi gerekli
     */
    public function requiresStudyField(): bool
    {
        return $this->gradeLevel >= 10 && $this->gradeLevel <= 12;
    }
}
