<?php
/**
 * ExamSession Model
 * 
 * Sınav oturumları: Tarih, saat ve şehir bazlı
 */

declare(strict_types=1);

namespace App\Models;

use DateTime;

class ExamSession
{
    public function __construct(
        public ?int $id = null,
        public int $examTypeId = 0,
        public int $cityId = 0,
        public ?DateTime $examDate = null,
        public ?string $examTime = null,
        public ?string $location = null,
        public ?int $capacity = null,
        public int $registeredCount = 0,
        public bool $isActive = true,
        public ?DateTime $registrationDeadline = null,
        public ?DateTime $createdAt = null,
        // İlişkili veriler
        public ?string $examTypeName = null,
        public ?string $cityName = null
    ) {}

    /**
     * Create ExamSession from database row
     */
    public static function fromArray(array $data): self
    {
        return new self(
            id: isset($data['id']) ? (int) $data['id'] : null,
            examTypeId: isset($data['exam_type_id']) ? (int) $data['exam_type_id'] : 0,
            cityId: isset($data['city_id']) ? (int) $data['city_id'] : 0,
            examDate: isset($data['exam_date']) ? new DateTime($data['exam_date']) : null,
            examTime: $data['exam_time'] ?? null,
            location: $data['location'] ?? null,
            capacity: isset($data['capacity']) ? (int) $data['capacity'] : null,
            registeredCount: isset($data['registered_count']) ? (int) $data['registered_count'] : 0,
            isActive: isset($data['is_active']) ? (bool) $data['is_active'] : true,
            registrationDeadline: isset($data['registration_deadline']) ? new DateTime($data['registration_deadline']) : null,
            createdAt: isset($data['created_at']) ? new DateTime($data['created_at']) : null,
            examTypeName: $data['exam_type_name'] ?? null,
            cityName: $data['city_name'] ?? null
        );
    }

    /**
     * Convert to array for API response
     */
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'exam_type_id' => $this->examTypeId,
            'city_id' => $this->cityId,
            'exam_date' => $this->examDate?->format('Y-m-d'),
            'exam_time' => $this->examTime,
            'location' => $this->location,
            'capacity' => $this->capacity,
            'registered_count' => $this->registeredCount,
            'is_active' => $this->isActive,
            'registration_deadline' => $this->registrationDeadline?->format('Y-m-d H:i:s'),
            'exam_type_name' => $this->examTypeName,
            'city_name' => $this->cityName,
            'display_text' => $this->getDisplayText(),
            'is_available' => $this->isAvailable(),
        ];
    }

    /**
     * Formda gösterilecek metin
     */
    public function getDisplayText(): string
    {
        $date = $this->examDate?->format('d.m.Y') ?? '';
        $time = $this->examTime ? substr($this->examTime, 0, 5) : '';
        $location = $this->location ? " - {$this->location}" : '';
        
        return "{$date} {$time}{$location}";
    }

    /**
     * Kayıt için müsait mi?
     */
    public function isAvailable(): bool
    {
        if (!$this->isActive) {
            return false;
        }

        // Kapasite kontrolü
        if ($this->capacity !== null && $this->registeredCount >= $this->capacity) {
            return false;
        }

        // Son kayıt tarihi kontrolü
        if ($this->registrationDeadline !== null && new DateTime() > $this->registrationDeadline) {
            return false;
        }

        // Sınav tarihi geçmiş mi?
        if ($this->examDate !== null && $this->examDate < new DateTime('today')) {
            return false;
        }

        return true;
    }

    /**
     * Kalan kontenjan
     */
    public function getRemainingCapacity(): ?int
    {
        if ($this->capacity === null) {
            return null;
        }
        return max(0, $this->capacity - $this->registeredCount);
    }
}
