<?php
/**
 * FormController
 * 
 * Handles all API endpoints for the registration form
 * Requirements: 1.2, 2.1, 3.2, 7.1
 */

declare(strict_types=1);

namespace App\Controllers;

use App\Models\Registration;
use App\Services\DatabaseService;
use App\Services\EncryptionService;
use App\Services\SchoolApiClient;
use App\Services\ValidationService;

class FormController
{
    private DatabaseService $databaseService;
    private ValidationService $validationService;
    private SchoolApiClient $schoolApiClient;
    private EncryptionService $encryptionService;

    public function __construct(
        ?DatabaseService $databaseService = null,
        ?ValidationService $validationService = null,
        ?SchoolApiClient $schoolApiClient = null,
        ?EncryptionService $encryptionService = null
    ) {
        $encryptionKey = $_ENV['ENCRYPTION_KEY'] ?? 'default-32-char-encryption-key!!';
        $this->encryptionService = $encryptionService ?? new EncryptionService($encryptionKey);
        $this->databaseService = $databaseService ?? new DatabaseService(null, $this->encryptionService);
        $this->validationService = $validationService ?? new ValidationService();
        $this->schoolApiClient = $schoolApiClient ?? new SchoolApiClient();
    }

    /**
     * Get all cities
     * 
     * GET /api/cities
     */
    public function getCities(): void
    {
        try {
            $cities = $this->databaseService->getCities();
            
            $response = array_map(fn($city) => [
                'id' => $city->id,
                'name' => $city->name,
                'code' => $city->code,
            ], $cities);

            $this->jsonResponse($response);
        } catch (\Exception $e) {
            $this->errorResponse('Şehirler yüklenirken bir hata oluştu', 500);
        }
    }

    /**
     * Get all active exam types
     * 
     * GET /api/exam-types
     */
    public function getExamTypes(): void
    {
        try {
            $examTypes = $this->databaseService->getExamTypes();
            
            $response = array_map(fn($examType) => $examType->toArray(), $examTypes);

            $this->jsonResponse($response);
        } catch (\Exception $e) {
            $this->errorResponse('Sınav türleri yüklenirken bir hata oluştu', 500);
        }
    }

    /**
     * Get exam sessions by city and exam type
     * 
     * GET /api/exam-sessions?cityId={cityId}&examTypeId={examTypeId}
     */
    public function getExamSessions(): void
    {
        try {
            $cityId = (int) ($_GET['cityId'] ?? 0);
            $examTypeId = isset($_GET['examTypeId']) ? (int) $_GET['examTypeId'] : null;

            if ($cityId <= 0) {
                $this->errorResponse('Geçersiz şehir ID', 400);
                return;
            }

            $sessions = $this->databaseService->getExamSessions($cityId, $examTypeId);
            
            $response = array_map(fn($session) => $session->toArray(), $sessions);

            $this->jsonResponse($response);
        } catch (\Exception $e) {
            $this->errorResponse('Sınav oturumları yüklenirken bir hata oluştu', 500);
        }
    }

    /**
     * Get grade levels by exam type
     * 
     * GET /api/grade-levels/{examTypeId}
     */
    public function getGradeLevels(string $examTypeId): void
    {
        try {
            $examTypeIdInt = (int) $examTypeId;
            
            if ($examTypeIdInt <= 0) {
                $this->errorResponse('Geçersiz sınav türü ID', 400);
                return;
            }

            $gradeLevels = $this->databaseService->getGradeLevelsByExamType($examTypeIdInt);
            
            $response = array_map(fn($level) => [
                'id' => $level->id,
                'examTypeId' => $level->examTypeId,
                'level' => $level->level,
                'displayName' => $level->displayName,
                'requiresStudyField' => $level->requiresStudyField(),
            ], $gradeLevels);

            $this->jsonResponse($response);
        } catch (\Exception $e) {
            $this->errorResponse('Sınıf düzeyleri yüklenirken bir hata oluştu', 500);
        }
    }

    /**
     * Search schools by city and query
     * 
     * GET /api/schools/search?cityId={cityId}&query={query}
     */
    public function searchSchools(): void
    {
        try {
            $cityId = (int) ($_GET['cityId'] ?? 0);
            $query = $_GET['query'] ?? '';

            if ($cityId <= 0) {
                $this->errorResponse('Geçersiz şehir ID', 400);
                return;
            }

            if (strlen($query) < 2) {
                $this->jsonResponse([]);
                return;
            }

            // Şehrin plaka kodunu veritabanından al
            $cities = $this->databaseService->getCities();
            foreach ($cities as $city) {
                if ($city->id === $cityId) {
                    $this->schoolApiClient->setPlateCodeFromDatabase($cityId, $city->code);
                    break;
                }
            }

            $schools = $this->schoolApiClient->searchSchools($cityId, $query);
            
            $response = array_map(fn($school) => [
                'id' => $school->id,
                'name' => $school->name,
                'fullName' => $school->getFullName(),
                'cityId' => $school->cityId,
                'type' => $school->type,
                'district' => $school->district,
            ], $schools);

            $this->jsonResponse($response);
        } catch (\Exception $e) {
            $this->errorResponse('Okullar aranırken bir hata oluştu: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Submit registration form
     * 
     * POST /api/registration
     */
    public function submitRegistration(): void
    {
        try {
            // Get JSON input
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input) {
                $this->errorResponse('Geçersiz istek verisi', 400);
                return;
            }

            // Sanitize inputs
            $data = $this->sanitizeRegistrationData($input);

            // Validate
            $validationResult = $this->validationService->validateRegistration($data);
            
            if (!$validationResult->isValid()) {
                $this->validationErrorResponse($validationResult->getErrors());
                return;
            }

            // Get school name from API
            $school = $this->schoolApiClient->getSchoolById($data['schoolId']);
            $schoolName = $school ? $school->name : ($data['schoolName'] ?? 'Bilinmeyen Okul');

            // Create registration model
            $registration = new Registration(
                cityId: $data['cityId'],
                examSessionId: $data['examSessionId'],
                examTypeId: $data['examTypeId'],
                gradeLevel: $data['gradeLevel'],
                studyField: $data['studyField'] ?? null,
                studentFirstName: $data['studentFirstName'],
                studentLastName: $data['studentLastName'],
                schoolId: $data['schoolId'],
                schoolName: $schoolName,
                parentFirstName: $data['parentFirstName'],
                parentLastName: $data['parentLastName'],
                parentPhone1: $this->validationService->normalizePhoneNumber($data['parentPhone1']),
                parentPhone2: isset($data['parentPhone2']) && !empty($data['parentPhone2']) 
                    ? $this->validationService->normalizePhoneNumber($data['parentPhone2']) 
                    : null,
                policyAccepted: $data['policyAccepted']
            );

            // Save to database
            $registrationId = $this->databaseService->saveRegistration($registration);

            // Update exam session registered count
            $this->databaseService->incrementExamSessionCount($data['examSessionId']);

            $this->jsonResponse([
                'success' => true,
                'message' => 'Kayıt başarıyla tamamlandı',
                'registrationId' => $registrationId,
            ], 201);
        } catch (\Exception $e) {
            $this->errorResponse('Kayıt sırasında bir hata oluştu: ' . $e->getMessage(), 500);
        }
    }

    // ============ Legacy endpoints (backward compatibility) ============

    /**
     * Get branches by city ID (Legacy - redirects to exam sessions)
     * @deprecated Use getExamSessions instead
     */
    public function getBranches(string $cityId): void
    {
        $_GET['cityId'] = $cityId;
        $this->getExamSessions();
    }

    /**
     * Sanitize registration data
     */
    private function sanitizeRegistrationData(array $input): array
    {
        return [
            'cityId' => isset($input['cityId']) ? (int) $input['cityId'] : 0,
            'examSessionId' => isset($input['examSessionId']) ? (int) $input['examSessionId'] : 0,
            'examTypeId' => isset($input['examTypeId']) ? (int) $input['examTypeId'] : 0,
            'gradeLevel' => isset($input['gradeLevel']) ? (int) $input['gradeLevel'] : 0,
            'studyField' => isset($input['studyField']) 
                ? $this->validationService->sanitizeInput($input['studyField']) 
                : null,
            'studentFirstName' => isset($input['studentFirstName']) 
                ? $this->validationService->sanitizeInput($input['studentFirstName']) 
                : '',
            'studentLastName' => isset($input['studentLastName']) 
                ? $this->validationService->sanitizeInput($input['studentLastName']) 
                : '',
            'schoolId' => isset($input['schoolId']) ? (int) $input['schoolId'] : 0,
            'schoolName' => isset($input['schoolName']) 
                ? $this->validationService->sanitizeInput($input['schoolName']) 
                : '',
            'parentFirstName' => isset($input['parentFirstName']) 
                ? $this->validationService->sanitizeInput($input['parentFirstName']) 
                : '',
            'parentLastName' => isset($input['parentLastName']) 
                ? $this->validationService->sanitizeInput($input['parentLastName']) 
                : '',
            'parentPhone1' => isset($input['parentPhone1']) 
                ? $this->validationService->sanitizeInput($input['parentPhone1']) 
                : '',
            'parentPhone2' => isset($input['parentPhone2']) && !empty($input['parentPhone2'])
                ? $this->validationService->sanitizeInput($input['parentPhone2']) 
                : null,
            'policyAccepted' => isset($input['policyAccepted']) ? (bool) $input['policyAccepted'] : false,
        ];
    }

    /**
     * Send JSON response
     */
    private function jsonResponse(mixed $data, int $statusCode = 200): void
    {
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Send error response
     */
    private function errorResponse(string $message, int $statusCode): void
    {
        $this->jsonResponse(['error' => $message], $statusCode);
    }

    /**
     * Send validation error response
     */
    private function validationErrorResponse(array $errors): void
    {
        $this->jsonResponse(['errors' => $errors], 422);
    }
}
