/**
 * Step Navigation Controller
 * Form adımları arasında geçişi yönetir
 */
const StepController = (function() {
    let currentStep = 1;
    const totalSteps = 3;

    function init() {
        updateUI();
        bindEvents();
    }

    function bindEvents() {
        document.getElementById('nextBtn')?.addEventListener('click', nextStep);
        document.getElementById('prevBtn')?.addEventListener('click', prevStep);
        document.getElementById('newRegistrationBtn')?.addEventListener('click', resetForm);
        document.getElementById('privacyPolicyLink')?.addEventListener('click', showPrivacyModal);
        document.getElementById('closePrivacyModal')?.addEventListener('click', hidePrivacyModal);
        document.getElementById('privacyModal')?.addEventListener('click', function(e) {
            if (e.target === this) hidePrivacyModal();
        });
    }

    function nextStep() {
        if (!validateCurrentStep()) return;
        
        if (currentStep < totalSteps) {
            currentStep++;
            updateUI();
        } else {
            submitForm();
        }
    }

    function prevStep() {
        if (currentStep > 1) {
            currentStep--;
            updateUI();
        }
    }

    function goToStep(step) {
        if (step >= 1 && step <= totalSteps) {
            currentStep = step;
            updateUI();
        }
    }

    function updateUI() {
        // Update sections
        document.querySelectorAll('.form-section').forEach(section => {
            section.classList.remove('active');
        });
        const activeSection = document.querySelector(`[data-section="${currentStep}"]`);
        if (activeSection) activeSection.classList.add('active');

        // Update progress steps
        document.querySelectorAll('.progress-step').forEach(step => {
            const stepNum = parseInt(step.dataset.step);
            step.classList.remove('active', 'completed');
            if (stepNum === currentStep) {
                step.classList.add('active');
            } else if (stepNum < currentStep) {
                step.classList.add('completed');
            }
        });

        // Update buttons
        const prevBtn = document.getElementById('prevBtn');
        const nextBtn = document.getElementById('nextBtn');

        prevBtn?.classList.toggle('hidden', currentStep === 1);
        
        if (currentStep === totalSteps) {
            nextBtn.innerHTML = `
                Kayıt Ol
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <polyline points="20 6 9 17 4 12"></polyline>
                </svg>
            `;
        } else {
            nextBtn.innerHTML = `
                İleri
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <polyline points="9 18 15 12 9 6"></polyline>
                </svg>
            `;
        }

        window.scrollTo({ top: 0, behavior: 'smooth' });
    }

    function validateCurrentStep() {
        let isValid = true;
        
        switch(currentStep) {
            case 1:
                isValid = validateStep1();
                break;
            case 2:
                isValid = validateStep2();
                break;
            case 3:
                isValid = validateStep3();
                break;
        }
        
        return isValid;
    }

    // Step 1: Sınav Seçimi (Şehir, Sınav)
    function validateStep1() {
        let valid = true;
        const city = document.getElementById('city');
        const examSession = document.getElementById('examSession');

        if (!city || !city.value) {
            showError('city', 'Şehir seçiniz');
            valid = false;
        } else {
            hideError('city');
        }

        if (!examSession || !examSession.value) {
            showError('examSession', 'Sınav seçiniz');
            valid = false;
        } else {
            hideError('examSession');
        }

        return valid;
    }

    // Step 2: Öğrenci Bilgileri (Ad, Soyad, Sınıf, Alan, Okul)
    function validateStep2() {
        let valid = true;
        const firstName = document.getElementById('studentFirstName');
        const lastName = document.getElementById('studentLastName');
        const gradeLevel = document.getElementById('gradeLevel');
        const studyField = document.getElementById('studyField');
        const studyFieldContainer = document.getElementById('studyFieldContainer');
        const schoolId = document.getElementById('schoolId');

        if (!firstName?.value.trim()) {
            showError('studentFirstName', 'Öğrenci adı giriniz');
            valid = false;
        } else {
            hideError('studentFirstName');
        }

        if (!lastName?.value.trim()) {
            showError('studentLastName', 'Öğrenci soyadı giriniz');
            valid = false;
        } else {
            hideError('studentLastName');
        }

        if (!gradeLevel?.value) {
            showError('gradeLevel', 'Sınıf düzeyi seçiniz');
            valid = false;
        } else {
            hideError('gradeLevel');
        }

        // Check study field for grades 10-12
        const level = parseInt(gradeLevel?.value);
        if (level >= 10 && level <= 12 && !studyFieldContainer?.classList.contains('hidden')) {
            if (!studyField?.value) {
                showError('studyField', 'Alan bilgisi seçiniz');
                valid = false;
            } else {
                hideError('studyField');
            }
        }

        if (!schoolId?.value) {
            showError('school', 'Okul seçiniz');
            valid = false;
        } else {
            hideError('school');
        }

        return valid;
    }

    // Step 3: Veli Bilgileri (Ad, Soyad, Telefonlar, Onay)
    function validateStep3() {
        let valid = true;
        const parentFirstName = document.getElementById('parentFirstName');
        const parentLastName = document.getElementById('parentLastName');
        const parentPhone1 = document.getElementById('parentPhone1');
        const policyAccepted = document.getElementById('policyAccepted');

        if (!parentFirstName?.value.trim()) {
            showError('parentFirstName', 'Veli adı giriniz');
            valid = false;
        } else {
            hideError('parentFirstName');
        }

        if (!parentLastName?.value.trim()) {
            showError('parentLastName', 'Veli soyadı giriniz');
            valid = false;
        } else {
            hideError('parentLastName');
        }

        const phone = parentPhone1?.value.replace(/\s/g, '');
        if (!phone || phone.length < 10) {
            showError('parentPhone1', 'Geçerli telefon numarası giriniz');
            valid = false;
        } else {
            hideError('parentPhone1');
        }

        if (!policyAccepted?.checked) {
            showError('policy', 'Gizlilik politikasını kabul etmelisiniz');
            valid = false;
        } else {
            hideError('policy');
        }

        return valid;
    }

    function showError(fieldId, message) {
        const errorEl = document.getElementById(fieldId + 'Error');
        if (errorEl) {
            errorEl.textContent = message;
            errorEl.classList.add('visible');
        }
    }

    function hideError(fieldId) {
        const errorEl = document.getElementById(fieldId + 'Error');
        if (errorEl) {
            errorEl.classList.remove('visible');
        }
    }

    async function submitForm() {
        const nextBtn = document.getElementById('nextBtn');
        nextBtn.disabled = true;
        nextBtn.innerHTML = '<div class="spinner"></div> Gönderiliyor...';

        try {
            const formData = {
                cityId: parseInt(document.getElementById('city').value),
                examSessionId: parseInt(document.getElementById('examSession').value),
                examTypeId: FormController.getState().examTypeId,
                gradeLevel: parseInt(document.getElementById('gradeLevel').value),
                studyField: document.getElementById('studyField')?.value || null,
                studentFirstName: document.getElementById('studentFirstName').value.trim(),
                studentLastName: document.getElementById('studentLastName').value.trim(),
                schoolId: parseInt(document.getElementById('schoolId').value),
                schoolName: document.getElementById('schoolName').value,
                parentFirstName: document.getElementById('parentFirstName').value.trim(),
                parentLastName: document.getElementById('parentLastName').value.trim(),
                parentPhone1: document.getElementById('parentPhone1').value.replace(/\s/g, ''),
                parentPhone2: document.getElementById('parentPhone2')?.value.replace(/\s/g, '') || null,
                policyAccepted: document.getElementById('policyAccepted').checked
            };

            const response = await APIClient.submitRegistration(formData);
            
            if (response.success) {
                showSuccess();
            }
        } catch (error) {
            console.error('Submit error:', error);
            alert('Kayıt sırasında bir hata oluştu. Lütfen tekrar deneyin.');
            nextBtn.disabled = false;
            nextBtn.innerHTML = `
                Kayıt Ol
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <polyline points="20 6 9 17 4 12"></polyline>
                </svg>
            `;
        }
    }

    function showSuccess() {
        document.querySelectorAll('.form-section').forEach(s => s.classList.remove('active'));
        document.getElementById('successScreen')?.classList.add('visible');
        document.getElementById('bottomBar')?.classList.add('hidden');
        document.querySelector('.progress-container')?.classList.add('hidden');
    }

    function resetForm() {
        document.getElementById('registrationForm')?.reset();
        document.getElementById('successScreen')?.classList.remove('visible');
        document.getElementById('bottomBar')?.classList.remove('hidden');
        document.querySelector('.progress-container')?.classList.remove('hidden');
        document.getElementById('selectedSchool')?.classList.remove('visible');
        document.getElementById('studyFieldContainer')?.classList.add('hidden');
        
        // Reset selects
        const examSession = document.getElementById('examSession');
        const gradeLevel = document.getElementById('gradeLevel');
        
        if (examSession) {
            examSession.disabled = true;
            examSession.innerHTML = '<option value="">Önce şehir seçiniz</option>';
        }
        if (gradeLevel) {
            gradeLevel.disabled = true;
            gradeLevel.innerHTML = '<option value="">Önce sınav seçiniz</option>';
        }
        
        currentStep = 1;
        updateUI();
        
        if (typeof FormController !== 'undefined') {
            FormController.init();
        }
    }

    function showPrivacyModal(e) {
        e.preventDefault();
        document.getElementById('privacyModal').style.display = 'flex';
    }

    function hidePrivacyModal() {
        document.getElementById('privacyModal').style.display = 'none';
    }

    return { init, goToStep, getCurrentStep: () => currentStep };
})();

document.addEventListener('DOMContentLoaded', function() {
    StepController.init();
});
