/**
 * SchoolAutocomplete Module
 * 
 * Implements school search autocomplete with debounce
 * Requirements: 3.2
 */

const SchoolAutocomplete = (function() {
    'use strict';

    // Configuration
    const DEBOUNCE_DELAY = 400; // milliseconds - MEB API için optimize edildi
    const MIN_QUERY_LENGTH = 3; // En az 3 karakter gerekli

    // State
    let debounceTimer = null;
    let currentQuery = '';
    let selectedIndex = -1;
    let schools = [];

    // DOM elements
    let elements = {};

    /**
     * Initialize the autocomplete
     * Requirements: 3.2
     */
    function init() {
        cacheElements();
        bindEvents();
    }

    /**
     * Cache DOM element references
     */
    function cacheElements() {
        elements = {
            input: document.getElementById('schoolSearch'),
            dropdown: document.getElementById('schoolDropdown'),
            loading: document.getElementById('schoolLoading'),
            schoolId: document.getElementById('schoolId'),
            schoolName: document.getElementById('schoolName'),
            selectedSchool: document.getElementById('selectedSchool'),
            selectedSchoolName: document.getElementById('selectedSchoolName'),
            citySelect: document.getElementById('city')
        };
    }

    /**
     * Bind event handlers
     */
    function bindEvents() {
        if (!elements.input) return;

        // Input event with debounce
        elements.input.addEventListener('input', onInput);

        // Keyboard navigation
        elements.input.addEventListener('keydown', onKeyDown);

        // Focus events
        elements.input.addEventListener('focus', onFocus);
        elements.input.addEventListener('blur', onBlur);

        // Click outside to close
        document.addEventListener('click', onDocumentClick);
    }

    /**
     * Handle input event with debounce
     * Requirements: 3.2
     */
    function onInput(e) {
        const query = e.target.value.trim();
        currentQuery = query;

        // Clear previous timer
        if (debounceTimer) {
            clearTimeout(debounceTimer);
        }

        // Clear selection if input changed
        if (elements.schoolId.value) {
            clearSelection();
        }

        // Check minimum query length
        if (query.length < MIN_QUERY_LENGTH) {
            hideDropdown();
            return;
        }

        // Debounce the search
        debounceTimer = setTimeout(() => {
            searchSchools(query);
        }, DEBOUNCE_DELAY);
    }

    /**
     * Search schools via API
     * Requirements: 3.2
     * Property 5: School Search Returns City-Filtered Results
     */
    async function searchSchools(query) {
        const cityId = parseInt(elements.citySelect?.value) || 0;

        if (!cityId) {
            showMessage('Önce şehir seçiniz');
            return;
        }

        if (query !== currentQuery) {
            return; // Query changed, skip this search
        }

        setLoading(true);

        try {
            schools = await APIClient.searchSchools(cityId, query);
            
            if (query !== currentQuery) {
                return; // Query changed during request
            }

            if (schools.length === 0) {
                showMessage('Sonuç bulunamadı');
            } else {
                renderResults(schools);
            }
        } catch (error) {
            console.error('School search error:', error);
            showMessage('Arama sırasında bir hata oluştu');
        } finally {
            setLoading(false);
        }
    }

    /**
     * Render search results in dropdown
     */
    function renderResults(results) {
        if (!elements.dropdown) return;

        selectedIndex = -1;
        
        const html = results.map((school, index) => {
            // fullName varsa onu kullan (ilçe dahil), yoksa sadece name
            const displayName = school.fullName || school.name;
            const district = school.district || '';
            const typeInfo = district ? `${district} - ${school.type}` : school.type;
            
            return `
                <div 
                    class="autocomplete-item"
                    data-index="${index}"
                    data-id="${school.id}"
                    data-name="${escapeHtml(displayName)}"
                    role="option"
                    aria-selected="false"
                >
                    <div class="autocomplete-item-name">${escapeHtml(school.name)}</div>
                    <div class="autocomplete-item-meta">${escapeHtml(typeInfo)}</div>
                </div>
            `;
        }).join('');

        elements.dropdown.innerHTML = html;
        showDropdown();

        // Bind click events to items
        elements.dropdown.querySelectorAll('.autocomplete-item').forEach(item => {
            item.addEventListener('click', () => {
                const index = parseInt(item.dataset.index);
                selectSchool(index);
            });
            
            item.addEventListener('mouseenter', () => {
                updateHighlight(parseInt(item.dataset.index));
            });
        });
    }

    /**
     * Show message in dropdown
     */
    function showMessage(message) {
        if (!elements.dropdown) return;

        elements.dropdown.innerHTML = `
            <div class="autocomplete-item">
                <div class="autocomplete-item-meta">${escapeHtml(message)}</div>
            </div>
        `;
        showDropdown();
    }

    /**
     * Handle keyboard navigation
     */
    function onKeyDown(e) {
        if (!isDropdownVisible()) {
            if (e.key === 'ArrowDown' && currentQuery.length >= MIN_QUERY_LENGTH) {
                searchSchools(currentQuery);
            }
            return;
        }

        switch (e.key) {
            case 'ArrowDown':
                e.preventDefault();
                navigateDown();
                break;
            case 'ArrowUp':
                e.preventDefault();
                navigateUp();
                break;
            case 'Enter':
                e.preventDefault();
                if (selectedIndex >= 0) {
                    selectSchool(selectedIndex);
                }
                break;
            case 'Escape':
                e.preventDefault();
                hideDropdown();
                break;
            case 'Tab':
                hideDropdown();
                break;
        }
    }

    /**
     * Navigate down in dropdown
     */
    function navigateDown() {
        if (schools.length === 0) return;
        
        const newIndex = selectedIndex < schools.length - 1 ? selectedIndex + 1 : 0;
        updateHighlight(newIndex);
    }

    /**
     * Navigate up in dropdown
     */
    function navigateUp() {
        if (schools.length === 0) return;
        
        const newIndex = selectedIndex > 0 ? selectedIndex - 1 : schools.length - 1;
        updateHighlight(newIndex);
    }

    /**
     * Update highlighted item
     */
    function updateHighlight(index) {
        const items = elements.dropdown?.querySelectorAll('.autocomplete-item');
        if (!items) return;

        // Remove previous highlight
        items.forEach(item => {
            item.classList.remove('highlighted');
            item.setAttribute('aria-selected', 'false');
        });

        // Add new highlight
        if (index >= 0 && index < items.length) {
            items[index].classList.add('highlighted');
            items[index].setAttribute('aria-selected', 'true');
            items[index].scrollIntoView({ block: 'nearest' });
        }

        selectedIndex = index;
    }

    /**
     * Select a school from results
     */
    function selectSchool(index) {
        if (index < 0 || index >= schools.length) return;

        const school = schools[index];
        const displayName = school.fullName || school.name;
        
        // Update hidden inputs
        elements.schoolId.value = school.id;
        elements.schoolName.value = displayName;
        
        // Update visible input
        elements.input.value = displayName;
        
        // Show selected indicator
        if (elements.selectedSchoolName) {
            elements.selectedSchoolName.textContent = displayName;
        }
        elements.selectedSchool?.classList.add('visible');
        
        // Update FormController state
        if (typeof FormController !== 'undefined') {
            FormController.setSchoolSelection(school.id, school.fullName || school.name);
        }
        
        hideDropdown();
    }

    /**
     * Clear school selection
     */
    function clearSelection() {
        elements.schoolId.value = '';
        elements.schoolName.value = '';
        elements.selectedSchool?.classList.remove('visible');
        
        if (typeof FormController !== 'undefined') {
            FormController.clearSchoolSelection();
        }
    }

    /**
     * Handle focus event
     */
    function onFocus() {
        if (currentQuery.length >= MIN_QUERY_LENGTH && schools.length > 0) {
            showDropdown();
        }
    }

    /**
     * Handle blur event
     */
    function onBlur() {
        // Delay to allow click on dropdown items
        setTimeout(() => {
            if (!elements.dropdown?.matches(':hover')) {
                hideDropdown();
            }
        }, 150);
    }

    /**
     * Handle document click
     */
    function onDocumentClick(e) {
        if (!elements.input?.contains(e.target) && 
            !elements.dropdown?.contains(e.target)) {
            hideDropdown();
        }
    }

    /**
     * Show dropdown
     */
    function showDropdown() {
        if (elements.dropdown) {
            elements.dropdown.classList.remove('hidden');
            elements.dropdown.classList.add('visible');
        }
        elements.input?.setAttribute('aria-expanded', 'true');
    }

    /**
     * Hide dropdown
     */
    function hideDropdown() {
        if (elements.dropdown) {
            elements.dropdown.classList.remove('visible');
            elements.dropdown.classList.add('hidden');
        }
        elements.input?.setAttribute('aria-expanded', 'false');
        selectedIndex = -1;
    }

    /**
     * Check if dropdown is visible
     */
    function isDropdownVisible() {
        return elements.dropdown?.classList.contains('visible');
    }

    /**
     * Set loading state
     */
    function setLoading(isLoading) {
        elements.loading?.classList.toggle('visible', isLoading);
    }

    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * Reset autocomplete state
     */
    function reset() {
        currentQuery = '';
        selectedIndex = -1;
        schools = [];
        
        if (elements.input) {
            elements.input.value = '';
        }
        
        clearSelection();
        hideDropdown();
    }

    // Public API
    return {
        init,
        reset,
        clearSelection,
        hideDropdown
    };
})();

// Export for module systems if available
if (typeof module !== 'undefined' && module.exports) {
    module.exports = SchoolAutocomplete;
}
