/**
 * PhoneMask Module
 * 
 * Implements Turkish phone number input masking (5XX XXX XX XX)
 * Requirements: 4.2
 */

const PhoneMask = (function() {
    'use strict';

    /**
     * Apply phone mask to an input element
     * Requirements: 4.2
     * 
     * @param {HTMLInputElement} input - Input element to apply mask to
     * @param {Function} onChange - Optional callback when value changes
     */
    function apply(input, onChange = null) {
        if (!input) return;

        // Handle input event
        input.addEventListener('input', function(e) {
            const cursorPosition = e.target.selectionStart;
            const oldValue = e.target.value;
            const newValue = formatValue(oldValue);
            
            e.target.value = newValue;
            
            // Adjust cursor position
            const newCursorPosition = calculateCursorPosition(oldValue, newValue, cursorPosition);
            e.target.setSelectionRange(newCursorPosition, newCursorPosition);
            
            // Call onChange callback if provided
            if (onChange) {
                onChange(getRawValue(newValue));
            }
        });

        // Handle paste event
        input.addEventListener('paste', function(e) {
            e.preventDefault();
            const pastedText = (e.clipboardData || window.clipboardData).getData('text');
            const cleanedText = pastedText.replace(/[^0-9]/g, '');
            
            // Only allow numbers starting with 5
            if (cleanedText.length > 0 && cleanedText[0] !== '5') {
                return;
            }
            
            const formattedValue = formatValue(cleanedText);
            e.target.value = formattedValue;
            
            if (onChange) {
                onChange(getRawValue(formattedValue));
            }
        });

        // Handle keydown to prevent invalid characters
        input.addEventListener('keydown', function(e) {
            // Allow: backspace, delete, tab, escape, enter, arrows
            if ([8, 9, 27, 13, 46, 37, 38, 39, 40].includes(e.keyCode)) {
                return;
            }
            
            // Allow: Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
            if ((e.ctrlKey || e.metaKey) && [65, 67, 86, 88].includes(e.keyCode)) {
                return;
            }
            
            // Ensure it's a number
            if ((e.shiftKey || (e.keyCode < 48 || e.keyCode > 57)) && 
                (e.keyCode < 96 || e.keyCode > 105)) {
                e.preventDefault();
                return;
            }
            
            // Check if first character must be 5
            const currentValue = e.target.value.replace(/\s/g, '');
            if (currentValue.length === 0) {
                const key = e.key || String.fromCharCode(e.keyCode);
                if (key !== '5') {
                    e.preventDefault();
                    return;
                }
            }
            
            // Limit to 10 digits
            if (currentValue.length >= 10) {
                e.preventDefault();
            }
        });

        // Handle focus - select all for easy replacement
        input.addEventListener('focus', function(e) {
            // Small delay to ensure selection works
            setTimeout(() => {
                if (e.target.value) {
                    e.target.select();
                }
            }, 0);
        });
    }

    /**
     * Format phone number value (5XX XXX XX XX)
     * 
     * @param {string} value - Raw input value
     * @returns {string} - Formatted value
     */
    function formatValue(value) {
        // Remove all non-digits
        let digits = value.replace(/[^0-9]/g, '');
        
        // Ensure first digit is 5
        if (digits.length > 0 && digits[0] !== '5') {
            digits = '';
        }
        
        // Limit to 10 digits
        digits = digits.slice(0, 10);
        
        // Format: 5XX XXX XX XX
        let formatted = '';
        
        if (digits.length > 0) {
            formatted = digits.slice(0, 3);
        }
        if (digits.length > 3) {
            formatted += ' ' + digits.slice(3, 6);
        }
        if (digits.length > 6) {
            formatted += ' ' + digits.slice(6, 8);
        }
        if (digits.length > 8) {
            formatted += ' ' + digits.slice(8, 10);
        }
        
        return formatted;
    }

    /**
     * Get raw value (digits only)
     * 
     * @param {string} formattedValue - Formatted phone number
     * @returns {string} - Raw digits
     */
    function getRawValue(formattedValue) {
        return formattedValue.replace(/\s/g, '');
    }

    /**
     * Calculate new cursor position after formatting
     * 
     * @param {string} oldValue - Value before formatting
     * @param {string} newValue - Value after formatting
     * @param {number} oldPosition - Cursor position before formatting
     * @returns {number} - New cursor position
     */
    function calculateCursorPosition(oldValue, newValue, oldPosition) {
        // Count digits before cursor in old value
        const digitsBeforeCursor = oldValue.slice(0, oldPosition).replace(/[^0-9]/g, '').length;
        
        // Find position in new value that has same number of digits before it
        let digitCount = 0;
        let newPosition = 0;
        
        for (let i = 0; i < newValue.length; i++) {
            if (/[0-9]/.test(newValue[i])) {
                digitCount++;
            }
            if (digitCount === digitsBeforeCursor) {
                newPosition = i + 1;
                break;
            }
        }
        
        // If we haven't found the position, put cursor at end
        if (digitCount < digitsBeforeCursor) {
            newPosition = newValue.length;
        }
        
        return newPosition;
    }

    /**
     * Validate if a phone number is complete and valid
     * 
     * @param {string} value - Phone number value
     * @returns {boolean}
     */
    function isValid(value) {
        const digits = getRawValue(value);
        return digits.length === 10 && digits[0] === '5';
    }

    /**
     * Initialize phone masks on all phone inputs
     */
    function initAll() {
        const phone1 = document.getElementById('parentPhone1');
        const phone2 = document.getElementById('parentPhone2');
        
        if (phone1) {
            apply(phone1, (value) => {
                // Update form state if FormController is available
                if (typeof FormController !== 'undefined') {
                    const state = FormController.getState();
                    state.parentPhone1 = value;
                }
            });
        }
        
        if (phone2) {
            apply(phone2, (value) => {
                // Update form state if FormController is available
                if (typeof FormController !== 'undefined') {
                    const state = FormController.getState();
                    state.parentPhone2 = value;
                }
            });
        }
    }

    // Public API
    return {
        apply,
        formatValue,
        getRawValue,
        isValid,
        initAll
    };
})();

// Export for module systems if available
if (typeof module !== 'undefined' && module.exports) {
    module.exports = PhoneMask;
}
