/**
 * FormValidation Module
 * 
 * Client-side validation for the registration form
 * Requirements: 4.2, 4.4
 */

const FormValidation = (function() {
    'use strict';

    // Turkish phone number regex (normalized: 5XXXXXXXXX - 10 digits starting with 5)
    const PHONE_REGEX_NORMALIZED = /^5[0-9]{9}$/;

    // Valid study fields
    const VALID_STUDY_FIELDS = ['sayisal', 'sozel', 'esit_agirlik', 'dil'];

    // Error messages in Turkish
    // Task 8.3: Turkish error messages
    const ERROR_MESSAGES = {
        // Field validation errors
        required: '{field} alanı zorunludur',
        invalidPhone: 'Geçerli bir telefon numarası giriniz (5XX XXX XX XX)',
        invalidStudyField: 'Geçerli bir alan seçiniz',
        policyRequired: 'Gizlilik politikasını kabul etmeniz gerekmektedir',
        
        // Additional validation errors
        tooShort: '{field} çok kısa',
        tooLong: '{field} çok uzun',
        invalidFormat: 'Geçersiz {field} formatı',
        
        // Network/Server errors
        networkError: 'Bağlantı hatası oluştu. Lütfen internet bağlantınızı kontrol edin.',
        timeoutError: 'İstek zaman aşımına uğradı. Lütfen tekrar deneyin.',
        serverError: 'Sunucu hatası oluştu. Lütfen daha sonra tekrar deneyin.',
        rateLimitError: 'Çok fazla istek gönderildi. Lütfen biraz bekleyip tekrar deneyin.',
        
        // Form submission errors
        submitError: 'Kayıt sırasında bir hata oluştu. Lütfen tekrar deneyin.',
        validationError: 'Lütfen formdaki hataları düzeltiniz',
        
        // Success messages
        submitSuccess: 'Kayıt başarıyla tamamlandı!'
    };

    // Field labels in Turkish
    const FIELD_LABELS = {
        cityId: 'Şehir',
        branchId: 'Şube',
        gradeLevel: 'Sınıf düzeyi',
        studyField: 'Alan bilgisi',
        studentFirstName: 'Öğrenci adı',
        studentLastName: 'Öğrenci soyadı',
        schoolId: 'Okul',
        parentFirstName: 'Veli adı',
        parentLastName: 'Veli soyadı',
        parentPhone1: 'Telefon numarası',
        parentPhone2: 'İkinci telefon numarası',
        policyAccepted: 'Gizlilik politikası'
    };

    /**
     * Validate the entire form
     * Requirements: 1.4, 3.3, 3.4, 4.3, 5.2
     * Property 6: Required Field Validation
     * Property 7: Conditional Study Field Validation
     * Property 8: Phone Number Format Validation
     * Property 9: Optional Phone2 Validation
     * 
     * @param {Object} state - Form state object
     * @returns {{isValid: boolean, errors: Object}}
     */
    function validateForm(state) {
        const errors = {};

        // Required field validation (Property 6)
        if (!state.cityId) {
            errors.city = formatError('required', 'cityId');
        }

        if (!state.branchId) {
            errors.branch = formatError('required', 'branchId');
        }

        if (!state.gradeLevel) {
            errors.gradeLevel = formatError('required', 'gradeLevel');
        }

        // Conditional study field validation (Property 7)
        if (state.gradeLevel >= 10 && state.gradeLevel <= 12) {
            if (!state.studyField || !VALID_STUDY_FIELDS.includes(state.studyField)) {
                errors.studyField = formatError('required', 'studyField');
            }
        }

        if (!state.studentFirstName || !state.studentFirstName.trim()) {
            errors.studentFirstName = formatError('required', 'studentFirstName');
        }

        if (!state.studentLastName || !state.studentLastName.trim()) {
            errors.studentLastName = formatError('required', 'studentLastName');
        }

        if (!state.schoolId) {
            errors.school = formatError('required', 'schoolId');
        }

        if (!state.parentFirstName || !state.parentFirstName.trim()) {
            errors.parentFirstName = formatError('required', 'parentFirstName');
        }

        if (!state.parentLastName || !state.parentLastName.trim()) {
            errors.parentLastName = formatError('required', 'parentLastName');
        }

        // Phone validation (Property 8)
        if (!state.parentPhone1 || !state.parentPhone1.trim()) {
            errors.parentPhone1 = formatError('required', 'parentPhone1');
        } else if (!validatePhoneNumber(state.parentPhone1)) {
            errors.parentPhone1 = ERROR_MESSAGES.invalidPhone;
        }

        // Optional phone2 validation (Property 9)
        if (state.parentPhone2 && state.parentPhone2.trim()) {
            if (!validatePhoneNumber(state.parentPhone2)) {
                errors.parentPhone2 = ERROR_MESSAGES.invalidPhone;
            }
        }

        // Policy acceptance validation
        if (!state.policyAccepted) {
            errors.policy = ERROR_MESSAGES.policyRequired;
        }

        return {
            isValid: Object.keys(errors).length === 0,
            errors
        };
    }

    /**
     * Validate a single phone number
     * Requirements: 4.2, 4.4
     * Property 8: Phone Number Format Validation
     * 
     * @param {string} phone - Phone number to validate
     * @returns {boolean}
     */
    function validatePhoneNumber(phone) {
        if (!phone) return false;
        
        // Remove all whitespace and normalize
        const normalized = normalizePhoneNumber(phone);
        
        return PHONE_REGEX_NORMALIZED.test(normalized);
    }

    /**
     * Normalize phone number (remove spaces and formatting)
     * 
     * @param {string} phone - Phone number to normalize
     * @returns {string}
     */
    function normalizePhoneNumber(phone) {
        if (!phone) return '';
        return phone.replace(/\s+/g, '').replace(/[^0-9]/g, '');
    }

    /**
     * Format phone number for display (5XX XXX XX XX)
     * 
     * @param {string} phone - Phone number to format
     * @returns {string}
     */
    function formatPhoneNumber(phone) {
        const normalized = normalizePhoneNumber(phone);
        
        if (normalized.length !== 10) {
            return phone;
        }
        
        return `${normalized.slice(0, 3)} ${normalized.slice(3, 6)} ${normalized.slice(6, 8)} ${normalized.slice(8, 10)}`;
    }

    /**
     * Validate study field
     * Property 7: Conditional Study Field Validation
     * 
     * @param {string} studyField - Study field value
     * @returns {boolean}
     */
    function validateStudyField(studyField) {
        return VALID_STUDY_FIELDS.includes(studyField);
    }

    /**
     * Check if study field is required based on grade level
     * Property 7: Conditional Study Field Validation
     * 
     * @param {number} gradeLevel - Grade level (1-12)
     * @returns {boolean}
     */
    function isStudyFieldRequired(gradeLevel) {
        return gradeLevel >= 10 && gradeLevel <= 12;
    }

    /**
     * Format error message with field label
     * 
     * @param {string} errorType - Error type key
     * @param {string} fieldKey - Field key for label lookup
     * @returns {string}
     */
    function formatError(errorType, fieldKey) {
        const template = ERROR_MESSAGES[errorType];
        const label = FIELD_LABELS[fieldKey] || fieldKey;
        return template.replace('{field}', label);
    }

    /**
     * Validate a single field
     * 
     * @param {string} fieldName - Field name
     * @param {any} value - Field value
     * @param {Object} context - Additional context (e.g., gradeLevel for studyField)
     * @returns {{isValid: boolean, error: string|null}}
     */
    function validateField(fieldName, value, context = {}) {
        let error = null;

        switch (fieldName) {
            case 'cityId':
            case 'branchId':
            case 'gradeLevel':
            case 'schoolId':
                if (!value) {
                    error = formatError('required', fieldName);
                }
                break;

            case 'studyField':
                if (isStudyFieldRequired(context.gradeLevel)) {
                    if (!value || !validateStudyField(value)) {
                        error = formatError('required', fieldName);
                    }
                }
                break;

            case 'studentFirstName':
            case 'studentLastName':
            case 'parentFirstName':
            case 'parentLastName':
                if (!value || !value.trim()) {
                    error = formatError('required', fieldName);
                }
                break;

            case 'parentPhone1':
                if (!value || !value.trim()) {
                    error = formatError('required', fieldName);
                } else if (!validatePhoneNumber(value)) {
                    error = ERROR_MESSAGES.invalidPhone;
                }
                break;

            case 'parentPhone2':
                // Optional, but if provided must be valid (Property 9)
                if (value && value.trim() && !validatePhoneNumber(value)) {
                    error = ERROR_MESSAGES.invalidPhone;
                }
                break;

            case 'policyAccepted':
                if (!value) {
                    error = ERROR_MESSAGES.policyRequired;
                }
                break;
        }

        return {
            isValid: error === null,
            error
        };
    }

    // Public API
    return {
        validateForm,
        validateField,
        validatePhoneNumber,
        normalizePhoneNumber,
        formatPhoneNumber,
        validateStudyField,
        isStudyFieldRequired,
        VALID_STUDY_FIELDS,
        ERROR_MESSAGES,
        FIELD_LABELS
    };
})();

// Export for module systems if available
if (typeof module !== 'undefined' && module.exports) {
    module.exports = FormValidation;
}
