/**
 * FormController Module
 * 
 * Manages form state, event handlers, and form flow
 */

const FormController = (function() {
    'use strict';

    // Form state
    let state = {
        cityId: null,
        examSessionId: null,
        examTypeId: null,
        gradeLevel: null,
        studyField: null,
        studentFirstName: '',
        studentLastName: '',
        schoolId: null,
        schoolName: '',
        parentFirstName: '',
        parentLastName: '',
        parentPhone1: '',
        parentPhone2: '',
        policyAccepted: false,
        loading: {
            cities: false,
            examSessions: false,
            gradeLevels: false,
            schools: false,
            submit: false
        },
        errors: {}
    };

    // Exam sessions cache (to get exam type from session)
    let examSessionsCache = [];

    // DOM element references
    let elements = {};

    /**
     * Initialize the form controller
     */
    async function init() {
        cacheElements();
        bindEvents();
        await loadCities();
    }

    /**
     * Cache DOM element references
     */
    function cacheElements() {
        elements = {
            form: document.getElementById('registrationForm'),
            // Selects
            city: document.getElementById('city'),
            examSession: document.getElementById('examSession'),
            gradeLevel: document.getElementById('gradeLevel'),
            studyField: document.getElementById('studyField'),
            studyFieldContainer: document.getElementById('studyFieldContainer'),
            // Inputs
            studentFirstName: document.getElementById('studentFirstName'),
            studentLastName: document.getElementById('studentLastName'),
            schoolSearch: document.getElementById('schoolSearch'),
            schoolId: document.getElementById('schoolId'),
            schoolName: document.getElementById('schoolName'),
            schoolDropdown: document.getElementById('schoolDropdown'),
            selectedSchool: document.getElementById('selectedSchool'),
            selectedSchoolName: document.getElementById('selectedSchoolName'),
            parentFirstName: document.getElementById('parentFirstName'),
            parentLastName: document.getElementById('parentLastName'),
            parentPhone1: document.getElementById('parentPhone1'),
            parentPhone2: document.getElementById('parentPhone2'),
            policyAccepted: document.getElementById('policyAccepted'),
            // Loading indicators
            cityLoading: document.getElementById('cityLoading'),
            examSessionLoading: document.getElementById('examSessionLoading'),
            gradeLevelLoading: document.getElementById('gradeLevelLoading'),
            schoolLoading: document.getElementById('schoolLoading'),
            loadingOverlay: document.getElementById('loadingOverlay'),
            // Error elements
            cityError: document.getElementById('cityError'),
            examSessionError: document.getElementById('examSessionError'),
            gradeLevelError: document.getElementById('gradeLevelError'),
            studyFieldError: document.getElementById('studyFieldError'),
            studentFirstNameError: document.getElementById('studentFirstNameError'),
            studentLastNameError: document.getElementById('studentLastNameError'),
            schoolError: document.getElementById('schoolError'),
            parentFirstNameError: document.getElementById('parentFirstNameError'),
            parentLastNameError: document.getElementById('parentLastNameError'),
            parentPhone1Error: document.getElementById('parentPhone1Error'),
            parentPhone2Error: document.getElementById('parentPhone2Error'),
            policyError: document.getElementById('policyError')
        };
    }

    /**
     * Bind event handlers
     */
    function bindEvents() {
        elements.city?.addEventListener('change', onCityChange);
        elements.examSession?.addEventListener('change', onExamSessionChange);
        elements.gradeLevel?.addEventListener('change', onGradeLevelChange);
        elements.studyField?.addEventListener('change', onStudyFieldChange);

        elements.studentFirstName?.addEventListener('input', onInputChange);
        elements.studentLastName?.addEventListener('input', onInputChange);
        elements.parentFirstName?.addEventListener('input', onInputChange);
        elements.parentLastName?.addEventListener('input', onInputChange);
        elements.policyAccepted?.addEventListener('change', onPolicyChange);

        elements.parentPhone1?.addEventListener('input', onPhoneChange);
        elements.parentPhone2?.addEventListener('input', onPhoneChange);

        document.addEventListener('click', (e) => {
            if (!elements.schoolSearch?.contains(e.target) && 
                !elements.schoolDropdown?.contains(e.target)) {
                hideSchoolDropdown();
            }
        });
    }

    /**
     * Load cities from API
     */
    async function loadCities() {
        setLoading('cities', true);
        try {
            const cities = await APIClient.getCities();
            
            // Check if no cities available (no active exams anywhere)
            if (cities.length === 0) {
                showEmptyState();
                return;
            }
            
            populateSelect(elements.city, cities, 'Şehir seçiniz');
        } catch (error) {
            console.error('Error loading cities:', error);
            showEmptyState();
        } finally {
            setLoading('cities', false);
        }
    }

    /**
     * Handle city selection change
     */
    async function onCityChange(e) {
        const cityId = parseInt(e.target.value) || null;
        
        state.cityId = cityId;
        state.examSessionId = null;
        state.examTypeId = null;
        state.gradeLevel = null;
        state.studyField = null;
        state.schoolId = null;
        state.schoolName = '';
        
        resetSelect(elements.examSession, 'Sınav seçiniz');
        resetSelect(elements.gradeLevel, 'Önce sınav seçiniz');
        hideStudyField();
        clearSchoolSelection();
        
        elements.examSession.disabled = !cityId;
        elements.gradeLevel.disabled = true;
        
        clearError('city');
        
        if (cityId) {
            await loadExamSessions(cityId);
        }
    }

    /**
     * Load exam sessions for selected city
     */
    async function loadExamSessions(cityId) {
        setLoading('examSessions', true);
        try {
            const sessions = await APIClient.getExamSessions(cityId);
            examSessionsCache = sessions;
            
            // Check if no sessions available
            if (sessions.length === 0) {
                showEmptyState();
                return;
            }
            
            // Hide empty state if it was shown
            hideEmptyState();
            
            // Format sessions for display
            const formattedSessions = sessions.map(s => ({
                id: s.id,
                name: `${s.exam_type_name} - ${s.display_text}`,
                examTypeId: s.exam_type_id
            }));
            
            populateSelect(elements.examSession, formattedSessions, 'Sınav seçiniz');
            elements.examSession.disabled = false;
        } catch (error) {
            console.error('Error loading exam sessions:', error);
        } finally {
            setLoading('examSessions', false);
        }
    }
    
    /**
     * Show empty state
     */
    function showEmptyState() {
        const emptyState = document.getElementById('emptyState');
        const form = document.getElementById('registrationForm');
        const bottomBar = document.querySelector('.bottom-bar');
        const mobileProgress = document.getElementById('mobileProgress');
        
        if (emptyState) emptyState.style.display = 'flex';
        if (form) form.style.display = 'none';
        if (bottomBar) bottomBar.style.display = 'none';
        if (mobileProgress) mobileProgress.style.display = 'none';
    }
    
    /**
     * Hide empty state
     */
    function hideEmptyState() {
        const emptyState = document.getElementById('emptyState');
        const form = document.getElementById('registrationForm');
        const bottomBar = document.querySelector('.bottom-bar');
        const mobileProgress = document.getElementById('mobileProgress');
        
        if (emptyState) emptyState.style.display = 'none';
        if (form) form.style.display = 'block';
        if (bottomBar) bottomBar.style.display = 'flex';
        if (mobileProgress) mobileProgress.style.display = 'block';
    }

    /**
     * Handle exam session selection change
     */
    async function onExamSessionChange(e) {
        const sessionId = parseInt(e.target.value) || null;
        
        state.examSessionId = sessionId;
        state.gradeLevel = null;
        state.studyField = null;
        
        resetSelect(elements.gradeLevel, 'Sınıf düzeyi seçiniz');
        hideStudyField();
        
        elements.gradeLevel.disabled = !sessionId;
        
        clearError('examSession');
        
        if (sessionId) {
            // Find exam type from cached sessions
            const session = examSessionsCache.find(s => s.id === sessionId);
            if (session) {
                state.examTypeId = session.exam_type_id;
                await loadGradeLevels(session.exam_type_id);
            }
        }
    }

    /**
     * Load grade levels for selected exam type
     */
    async function loadGradeLevels(examTypeId) {
        setLoading('gradeLevels', true);
        try {
            const gradeLevels = await APIClient.getGradeLevels(examTypeId);
            populateSelect(elements.gradeLevel, gradeLevels, 'Sınıf düzeyi seçiniz', 'level', 'displayName');
            elements.gradeLevel.disabled = false;
        } catch (error) {
            console.error('Error loading grade levels:', error);
        } finally {
            setLoading('gradeLevels', false);
        }
    }

    /**
     * Handle grade level selection change
     */
    function onGradeLevelChange(e) {
        const gradeLevel = parseInt(e.target.value) || null;
        
        state.gradeLevel = gradeLevel;
        state.studyField = null;
        
        clearError('gradeLevel');
        
        if (gradeLevel && gradeLevel >= 10 && gradeLevel <= 12) {
            showStudyField();
        } else {
            hideStudyField();
        }
    }

    function showStudyField() {
        elements.studyFieldContainer?.classList.remove('hidden');
        if (elements.studyField) elements.studyField.value = '';
        state.studyField = null;
    }

    function hideStudyField() {
        elements.studyFieldContainer?.classList.add('hidden');
        if (elements.studyField) elements.studyField.value = '';
        state.studyField = null;
        clearError('studyField');
    }

    function onStudyFieldChange(e) {
        state.studyField = e.target.value || null;
        clearError('studyField');
    }

    function onInputChange(e) {
        const field = e.target.id;
        const value = e.target.value;
        
        const fieldMap = {
            'studentFirstName': 'studentFirstName',
            'studentLastName': 'studentLastName',
            'parentFirstName': 'parentFirstName',
            'parentLastName': 'parentLastName'
        };
        
        if (fieldMap[field]) {
            state[fieldMap[field]] = value;
            clearError(field);
        }
    }

    function onPolicyChange(e) {
        state.policyAccepted = e.target.checked;
        clearError('policy');
    }

    function onPhoneChange(e) {
        const field = e.target.id;
        const value = e.target.value;
        
        if (field === 'parentPhone1') {
            state.parentPhone1 = value;
            clearError('parentPhone1');
        } else if (field === 'parentPhone2') {
            state.parentPhone2 = value;
            clearError('parentPhone2');
        }
    }

    function setLoading(type, isLoading) {
        state.loading[type] = isLoading;
        
        const loadingIndicators = {
            cities: elements.cityLoading,
            examSessions: elements.examSessionLoading,
            gradeLevels: elements.gradeLevelLoading,
            schools: elements.schoolLoading
        };
        
        const indicator = loadingIndicators[type];
        if (indicator) {
            indicator.classList.toggle('visible', isLoading);
        }
    }

    function populateSelect(select, items, placeholder, valueKey = 'id', textKey = 'name') {
        if (!select) return;
        
        select.innerHTML = `<option value="">${placeholder}</option>`;
        
        items.forEach(item => {
            const option = document.createElement('option');
            option.value = item[valueKey];
            option.textContent = item[textKey];
            select.appendChild(option);
        });
    }

    function resetSelect(select, placeholder) {
        if (!select) return;
        select.innerHTML = `<option value="">${placeholder}</option>`;
    }

    function clearSchoolSelection() {
        if (elements.schoolSearch) elements.schoolSearch.value = '';
        if (elements.schoolId) elements.schoolId.value = '';
        if (elements.schoolName) elements.schoolName.value = '';
        elements.selectedSchool?.classList.remove('visible');
        state.schoolId = null;
        state.schoolName = '';
    }

    function hideSchoolDropdown() {
        elements.schoolDropdown?.classList.add('hidden');
        elements.schoolDropdown?.classList.remove('visible');
    }

    function clearError(field) {
        const errorElement = elements[`${field}Error`];
        if (errorElement) {
            errorElement.textContent = '';
            errorElement.classList.remove('visible');
        }
    }

    function getState() {
        return { ...state };
    }

    function getCityId() {
        return state.cityId;
    }

    function setSchoolSelection(schoolId, schoolName) {
        state.schoolId = schoolId;
        state.schoolName = schoolName;
        
        if (elements.schoolId) elements.schoolId.value = schoolId;
        if (elements.schoolName) elements.schoolName.value = schoolName;
        if (elements.selectedSchoolName) elements.selectedSchoolName.textContent = schoolName;
        elements.selectedSchool?.classList.add('visible');
        
        clearError('school');
    }

    return { 
        init, 
        getState, 
        getCityId,
        clearSchoolSelection,
        hideSchoolDropdown,
        setSchoolSelection
    };
})();

document.addEventListener('DOMContentLoaded', function() {
    FormController.init();
});

if (typeof module !== 'undefined' && module.exports) {
    module.exports = FormController;
}
