/**
 * APIClient Module
 * 
 * Handles all API communication for the registration form
 */

const APIClient = (function() {
    'use strict';

    const BASE_URL = '/api';
    const TIMEOUT = 30000;

    /**
     * Make an HTTP request with timeout support
     */
    async function request(url, options = {}) {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), TIMEOUT);

        try {
            const response = await fetch(url, {
                ...options,
                signal: controller.signal,
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    ...options.headers
                }
            });

            clearTimeout(timeoutId);

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                const error = new Error(errorData.error || `HTTP ${response.status}`);
                error.status = response.status;
                error.data = errorData;
                throw error;
            }

            return await response.json();
        } catch (error) {
            clearTimeout(timeoutId);
            
            if (error.name === 'AbortError') {
                const timeoutError = new Error('İstek zaman aşımına uğradı');
                timeoutError.isTimeout = true;
                throw timeoutError;
            }
            
            if (!error.status) {
                error.isNetworkError = true;
                error.message = 'Bağlantı hatası oluştu';
            }
            
            throw error;
        }
    }

    /**
     * Get all cities
     */
    async function getCities() {
        return await request(`${BASE_URL}/cities`);
    }

    /**
     * Get exam sessions by city (and optionally exam type)
     */
    async function getExamSessions(cityId, examTypeId = null) {
        if (!cityId || cityId <= 0) {
            throw new Error('Geçersiz şehir ID');
        }
        let url = `${BASE_URL}/exam-sessions?cityId=${cityId}`;
        if (examTypeId) {
            url += `&examTypeId=${examTypeId}`;
        }
        return await request(url);
    }

    /**
     * Get grade levels by exam type ID
     */
    async function getGradeLevels(examTypeId) {
        if (!examTypeId || examTypeId <= 0) {
            throw new Error('Geçersiz sınav türü ID');
        }
        return await request(`${BASE_URL}/grade-levels/${examTypeId}`);
    }

    /**
     * Search schools by city and query
     */
    async function searchSchools(cityId, query) {
        if (!cityId || cityId <= 0) {
            throw new Error('Geçersiz şehir ID');
        }
        if (!query || query.length < 2) {
            return [];
        }
        const params = new URLSearchParams({ cityId: cityId.toString(), query });
        return await request(`${BASE_URL}/schools/search?${params}`);
    }

    /**
     * Submit registration form
     */
    async function submitRegistration(data) {
        return await request(`${BASE_URL}/registration`, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    }

    // Public API
    return {
        getCities,
        getExamSessions,
        getGradeLevels,
        searchSchools,
        submitRegistration
    };
})();

if (typeof module !== 'undefined' && module.exports) {
    module.exports = APIClient;
}
