<?php
/**
 * Registrations API
 * Update registration status
 */

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../../../vendor/autoload.php';

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/../../..');
$dotenv->safeLoad();

$config = require __DIR__ . '/../../../config/database.php';

try {
    $dsn = sprintf('%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'], $config['host'], $config['port'], 
        $config['database'], $config['charset']
    );
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Get the actual HTTP method (handle method override)
    $method = $_SERVER['REQUEST_METHOD'];
    
    // Check for method override in headers or body
    if ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        if (isset($input['_method'])) {
            $method = strtoupper($input['_method']);
        } elseif (isset($_SERVER['HTTP_X_HTTP_METHOD_OVERRIDE'])) {
            $method = strtoupper($_SERVER['HTTP_X_HTTP_METHOD_OVERRIDE']);
        }
    }
    
    if ($method === 'PUT') {
        handlePut($pdo);
    } elseif ($method === 'DELETE') {
        handleDelete($pdo);
    } else {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed: ' . $method]);
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}

function handlePut(PDO $pdo): void
{
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || empty($data['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing registration ID']);
        return;
    }
    
    $stmt = $pdo->prepare('
        UPDATE registrations 
        SET can_take_exam = :can_take_exam
        WHERE id = :id
    ');
    
    $stmt->execute([
        'id' => $data['id'],
        'can_take_exam' => isset($data['can_take_exam']) ? (int)$data['can_take_exam'] : 1,
    ]);
    
    echo json_encode(['success' => true, 'message' => 'Durum güncellendi']);
}

function handleDelete(PDO $pdo): void
{
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || empty($data['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing registration ID']);
        return;
    }
    
    $stmt = $pdo->prepare('DELETE FROM registrations WHERE id = :id');
    $stmt->execute(['id' => $data['id']]);
    
    echo json_encode(['success' => true, 'message' => 'Kayıt silindi']);
}
