<?php
/**
 * Exam Sessions API
 * CRUD operations for exam sessions
 */

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../../../vendor/autoload.php';

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/../../..');
$dotenv->safeLoad();

$config = require __DIR__ . '/../../../config/database.php';

try {
    $dsn = sprintf('%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'], $config['host'], $config['port'], 
        $config['database'], $config['charset']
    );
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Get the actual HTTP method (handle method override)
    $method = $_SERVER['REQUEST_METHOD'];
    
    // Check for method override in headers or body
    if ($method === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        if (isset($input['_method'])) {
            $method = strtoupper($input['_method']);
        } elseif (isset($_SERVER['HTTP_X_HTTP_METHOD_OVERRIDE'])) {
            $method = strtoupper($_SERVER['HTTP_X_HTTP_METHOD_OVERRIDE']);
        }
    }
    
    switch ($method) {
        case 'GET':
            handleGet($pdo);
            break;
        case 'POST':
            handlePost($pdo);
            break;
        case 'PUT':
            handlePut($pdo);
            break;
        case 'DELETE':
            handleDelete($pdo);
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed: ' . $method]);
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}

function handleGet(PDO $pdo): void
{
    if (isset($_GET['id'])) {
        // Get single session
        $stmt = $pdo->prepare('
            SELECT es.*, et.name as exam_type_name, c.name as city_name
            FROM exam_sessions es
            JOIN exam_types et ON es.exam_type_id = et.id
            JOIN cities c ON es.city_id = c.id
            WHERE es.id = :id
        ');
        $stmt->execute(['id' => $_GET['id']]);
        $session = $stmt->fetch();
        
        if ($session) {
            echo json_encode($session);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Session not found']);
        }
    } else {
        // Get all sessions
        $stmt = $pdo->query('
            SELECT es.*, et.name as exam_type_name, c.name as city_name,
                   COUNT(r.id) as registration_count
            FROM exam_sessions es
            JOIN exam_types et ON es.exam_type_id = et.id
            JOIN cities c ON es.city_id = c.id
            LEFT JOIN registrations r ON es.id = r.exam_session_id
            GROUP BY es.id
            ORDER BY es.exam_date DESC, es.exam_time DESC
        ');
        echo json_encode($stmt->fetchAll());
    }
}

function handlePost(PDO $pdo): void
{
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        $data = $_POST;
    }
    
    // Validate required fields
    $required = ['exam_type_id', 'city_id', 'exam_date', 'exam_time'];
    foreach ($required as $field) {
        if (empty($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    $stmt = $pdo->prepare('
        INSERT INTO exam_sessions (exam_type_id, city_id, exam_date, exam_time, location, capacity, is_active)
        VALUES (:exam_type_id, :city_id, :exam_date, :exam_time, :location, :capacity, 1)
    ');
    
    $stmt->execute([
        'exam_type_id' => $data['exam_type_id'],
        'city_id' => $data['city_id'],
        'exam_date' => $data['exam_date'],
        'exam_time' => $data['exam_time'],
        'location' => $data['location'] ?? null,
        'capacity' => !empty($data['capacity']) ? (int)$data['capacity'] : null,
    ]);
    
    $id = $pdo->lastInsertId();
    
    http_response_code(201);
    echo json_encode(['success' => true, 'id' => $id, 'message' => 'Sınav oturumu oluşturuldu']);
}

function handlePut(PDO $pdo): void
{
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || empty($data['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing session ID']);
        return;
    }
    
    $stmt = $pdo->prepare('
        UPDATE exam_sessions 
        SET exam_type_id = :exam_type_id,
            city_id = :city_id,
            exam_date = :exam_date,
            exam_time = :exam_time,
            location = :location,
            capacity = :capacity,
            is_active = :is_active
        WHERE id = :id
    ');
    
    $stmt->execute([
        'id' => $data['id'],
        'exam_type_id' => $data['exam_type_id'],
        'city_id' => $data['city_id'],
        'exam_date' => $data['exam_date'],
        'exam_time' => $data['exam_time'],
        'location' => $data['location'] ?? null,
        'capacity' => !empty($data['capacity']) ? (int)$data['capacity'] : null,
        'is_active' => isset($data['is_active']) ? (int)$data['is_active'] : 1,
    ]);
    
    echo json_encode(['success' => true, 'message' => 'Sınav oturumu güncellendi']);
}

function handleDelete(PDO $pdo): void
{
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || empty($data['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing session ID']);
        return;
    }
    
    // Check if there are registrations
    $stmt = $pdo->prepare('SELECT COUNT(*) FROM registrations WHERE exam_session_id = :id');
    $stmt->execute(['id' => $data['id']]);
    $count = $stmt->fetchColumn();
    
    if ($count > 0) {
        // Don't delete, just deactivate
        $stmt = $pdo->prepare('UPDATE exam_sessions SET is_active = 0 WHERE id = :id');
        $stmt->execute(['id' => $data['id']]);
        echo json_encode(['success' => true, 'message' => 'Sınav oturumu pasif edildi (kayıtlar var)']);
    } else {
        // Safe to delete
        $stmt = $pdo->prepare('DELETE FROM exam_sessions WHERE id = :id');
        $stmt->execute(['id' => $data['id']]);
        echo json_encode(['success' => true, 'message' => 'Sınav oturumu silindi']);
    }
}
