<?php
/**
 * Database Seeder
 * 
 * Usage: php database/seed.php
 */

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

// Load environment variables
$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->safeLoad();

// Load database configuration
$config = require __DIR__ . '/../config/database.php';

try {
    // Create PDO connection
    $dsn = sprintf(
        '%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'],
        $config['host'],
        $config['port'],
        $config['database'],
        $config['charset']
    );
    
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    echo "Connected to database: {$config['database']}\n\n";
    
    // Seed cities (81 il with plate codes)
    echo "Seeding cities...\n";
    $cities = [
        [1, 'Adana', '01'],
        [2, 'Adıyaman', '02'],
        [3, 'Afyonkarahisar', '03'],
        [4, 'Ağrı', '04'],
        [5, 'Amasya', '05'],
        [6, 'Ankara', '06'],
        [7, 'Antalya', '07'],
        [8, 'Artvin', '08'],
        [9, 'Aydın', '09'],
        [10, 'Balıkesir', '10'],
        [11, 'Bilecik', '11'],
        [12, 'Bingöl', '12'],
        [13, 'Bitlis', '13'],
        [14, 'Bolu', '14'],
        [15, 'Burdur', '15'],
        [16, 'Bursa', '16'],
        [17, 'Çanakkale', '17'],
        [18, 'Çankırı', '18'],
        [19, 'Çorum', '19'],
        [20, 'Denizli', '20'],
        [21, 'Diyarbakır', '21'],
        [22, 'Edirne', '22'],
        [23, 'Elazığ', '23'],
        [24, 'Erzincan', '24'],
        [25, 'Erzurum', '25'],
        [26, 'Eskişehir', '26'],
        [27, 'Gaziantep', '27'],
        [28, 'Giresun', '28'],
        [29, 'Gümüşhane', '29'],
        [30, 'Hakkari', '30'],
        [31, 'Hatay', '31'],
        [32, 'Isparta', '32'],
        [33, 'Mersin', '33'],
        [34, 'İstanbul', '34'],
        [35, 'İzmir', '35'],
        [36, 'Kars', '36'],
        [37, 'Kastamonu', '37'],
        [38, 'Kayseri', '38'],
        [39, 'Kırklareli', '39'],
        [40, 'Kırşehir', '40'],
        [41, 'Kocaeli', '41'],
        [42, 'Konya', '42'],
        [43, 'Kütahya', '43'],
        [44, 'Malatya', '44'],
        [45, 'Manisa', '45'],
        [46, 'Kahramanmaraş', '46'],
        [47, 'Mardin', '47'],
        [48, 'Muğla', '48'],
        [49, 'Muş', '49'],
        [50, 'Nevşehir', '50'],
        [51, 'Niğde', '51'],
        [52, 'Ordu', '52'],
        [53, 'Rize', '53'],
        [54, 'Sakarya', '54'],
        [55, 'Samsun', '55'],
        [56, 'Siirt', '56'],
        [57, 'Sinop', '57'],
        [58, 'Sivas', '58'],
        [59, 'Tekirdağ', '59'],
        [60, 'Tokat', '60'],
        [61, 'Trabzon', '61'],
        [62, 'Tunceli', '62'],
        [63, 'Şanlıurfa', '63'],
        [64, 'Uşak', '64'],
        [65, 'Van', '65'],
        [66, 'Yozgat', '66'],
        [67, 'Zonguldak', '67'],
        [68, 'Aksaray', '68'],
        [69, 'Bayburt', '69'],
        [70, 'Karaman', '70'],
        [71, 'Kırıkkale', '71'],
        [72, 'Batman', '72'],
        [73, 'Şırnak', '73'],
        [74, 'Bartın', '74'],
        [75, 'Ardahan', '75'],
        [76, 'Iğdır', '76'],
        [77, 'Yalova', '77'],
        [78, 'Karabük', '78'],
        [79, 'Kilis', '79'],
        [80, 'Osmaniye', '80'],
        [81, 'Düzce', '81'],
    ];
    
    $stmt = $pdo->prepare("INSERT IGNORE INTO cities (id, name, code, is_active) VALUES (?, ?, ?, TRUE)");
    foreach ($cities as $city) {
        $stmt->execute($city);
    }
    echo "  - " . count($cities) . " cities seeded\n";
    
    // Seed exam types
    echo "Seeding exam types...\n";
    $examTypes = [
        [1, 'YKS Deneme Sınavı', 'Yükseköğretim Kurumları Sınavı hazırlık deneme sınavı'],
        [2, 'LGS Deneme Sınavı', 'Liselere Geçiş Sınavı hazırlık deneme sınavı'],
        [3, 'TYT Deneme Sınavı', 'Temel Yeterlilik Testi deneme sınavı'],
        [4, 'AYT Deneme Sınavı', 'Alan Yeterlilik Testi deneme sınavı'],
    ];
    
    $stmt = $pdo->prepare("INSERT IGNORE INTO exam_types (id, name, description, is_active) VALUES (?, ?, ?, TRUE)");
    foreach ($examTypes as $examType) {
        $stmt->execute($examType);
    }
    echo "  - " . count($examTypes) . " exam types seeded\n";
    
    // Seed exam type grade levels
    echo "Seeding exam type grade levels...\n";
    $examTypeGradeLevels = [
        // YKS: 9, 10, 11, 12
        [1, 9, '9. Sınıf'],
        [1, 10, '10. Sınıf'],
        [1, 11, '11. Sınıf'],
        [1, 12, '12. Sınıf'],
        // LGS: 8
        [2, 8, '8. Sınıf'],
        // TYT: 11, 12
        [3, 11, '11. Sınıf'],
        [3, 12, '12. Sınıf'],
        // AYT: 11, 12
        [4, 11, '11. Sınıf'],
        [4, 12, '12. Sınıf'],
    ];
    
    $stmt = $pdo->prepare("INSERT IGNORE INTO exam_type_grade_levels (exam_type_id, grade_level, display_name) VALUES (?, ?, ?)");
    foreach ($examTypeGradeLevels as $level) {
        $stmt->execute($level);
    }
    echo "  - " . count($examTypeGradeLevels) . " exam type grade levels seeded\n";
    
    // Seed exam sessions (örnek veriler)
    echo "Seeding exam sessions...\n";
    $examSessions = [
        // Ankara YKS
        [1, 6, '2025-01-15', '10:00:00', 'Ankara Merkez Şube', 100],
        [1, 6, '2025-01-22', '10:00:00', 'Ankara Merkez Şube', 100],
        [1, 6, '2025-02-05', '14:00:00', 'Ankara Çankaya Şube', 80],
        // İstanbul YKS
        [1, 34, '2025-01-16', '10:00:00', 'İstanbul Kadıköy Şube', 150],
        [1, 34, '2025-01-23', '10:00:00', 'İstanbul Beşiktaş Şube', 120],
        // İzmir YKS
        [1, 35, '2025-01-17', '10:00:00', 'İzmir Konak Şube', 80],
        // Ankara LGS
        [2, 6, '2025-01-18', '09:00:00', 'Ankara Merkez Şube', 100],
        [2, 6, '2025-01-25', '09:00:00', 'Ankara Merkez Şube', 100],
        // İstanbul LGS
        [2, 34, '2025-01-19', '09:00:00', 'İstanbul Kadıköy Şube', 120],
        // TYT
        [3, 6, '2025-02-01', '10:00:00', 'Ankara Merkez Şube', 100],
        [3, 34, '2025-02-02', '10:00:00', 'İstanbul Kadıköy Şube', 120],
    ];
    
    $stmt = $pdo->prepare("INSERT IGNORE INTO exam_sessions (exam_type_id, city_id, exam_date, exam_time, location, capacity, is_active) VALUES (?, ?, ?, ?, ?, ?, TRUE)");
    foreach ($examSessions as $session) {
        $stmt->execute($session);
    }
    echo "  - " . count($examSessions) . " exam sessions seeded\n";
    
    echo "\nAll seed data inserted successfully!\n";
    
} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage() . "\n";
    exit(1);
}
