<?php
/**
 * Database Migration Runner
 * 
 * Usage: php database/migrate.php
 * 
 * Requirements: 7.1
 */

declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';

// Load environment variables
$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->safeLoad();

// Load database configuration
$config = require __DIR__ . '/../config/database.php';

try {
    // Create PDO connection
    $dsn = sprintf(
        '%s:host=%s;port=%s;charset=%s',
        $config['driver'],
        $config['host'],
        $config['port'],
        $config['charset']
    );
    
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Create database if not exists
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$config['database']}` 
                CHARACTER SET {$config['charset']} 
                COLLATE {$config['collation']}");
    
    $pdo->exec("USE `{$config['database']}`");
    
    echo "Connected to database: {$config['database']}\n";
    
    // Get migration files
    $migrationsPath = __DIR__ . '/migrations';
    $files = glob($migrationsPath . '/*.sql');
    sort($files);
    
    foreach ($files as $file) {
        $filename = basename($file);
        echo "Running migration: {$filename}... ";
        
        $sql = file_get_contents($file);
        
        try {
            $pdo->exec($sql);
            echo "Done\n";
        } catch (PDOException $e) {
            // Check if it's a "table already exists" error
            if (strpos($e->getMessage(), '1050') !== false) {
                echo "Skipped (table already exists)\n";
            } else {
                throw $e;
            }
        }
    }
    
    echo "\nAll migrations completed successfully!\n";
    
} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage() . "\n";
    exit(1);
}
