<?php
/**
 * Registrations List
 */

declare(strict_types=1);

require_once __DIR__ . '/components/auth.php';

require_once __DIR__ . '/../vendor/autoload.php';

use App\Services\EncryptionService;

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->safeLoad();

$config = require __DIR__ . '/../config/database.php';

// Initialize encryption service
$encryptionService = EncryptionService::fromConfig();

try {
    $dsn = sprintf('%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'], $config['host'], $config['port'], 
        $config['database'], $config['charset']
    );
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Filters
    $cityFilter = $_GET['city'] ?? '';
    $examTypeFilter = $_GET['exam_type'] ?? '';
    $dateFrom = $_GET['date_from'] ?? '';
    $dateTo = $_GET['date_to'] ?? '';
    $search = trim($_GET['search'] ?? '');
    $sessionFilter = $_GET['session_id'] ?? '';
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $perPage = 50;
    $offset = ($page - 1) * $perPage;
    
    // Build query
    $sql = '
        SELECT r.*, 
               c.name as city_name, 
               et.name as exam_type_name, 
               es.exam_date, 
               es.exam_time,
               es.location
        FROM registrations r
        JOIN cities c ON r.city_id = c.id
        JOIN exam_types et ON r.exam_type_id = et.id
        JOIN exam_sessions es ON r.exam_session_id = es.id
        WHERE 1=1
    ';
    
    $params = [];
    
    if ($cityFilter) {
        $sql .= ' AND r.city_id = :city_id';
        $params['city_id'] = $cityFilter;
    }
    
    if ($examTypeFilter) {
        $sql .= ' AND r.exam_type_id = :exam_type_id';
        $params['exam_type_id'] = $examTypeFilter;
    }
    
    if ($sessionFilter) {
        $sql .= ' AND r.exam_session_id = :session_id';
        $params['session_id'] = $sessionFilter;
    }
    
    if ($dateFrom) {
        $sql .= ' AND r.created_at >= :date_from';
        $params['date_from'] = $dateFrom . ' 00:00:00';
    }
    
    if ($dateTo) {
        $sql .= ' AND r.created_at <= :date_to';
        $params['date_to'] = $dateTo . ' 23:59:59';
    }
    
    if ($search !== '') {
        $sql .= ' AND (r.student_first_name LIKE :search1 
                    OR r.student_last_name LIKE :search2 
                    OR r.parent_first_name LIKE :search3 
                    OR r.parent_last_name LIKE :search4
                    OR r.school_name LIKE :search5)';
        $params['search1'] = '%' . $search . '%';
        $params['search2'] = '%' . $search . '%';
        $params['search3'] = '%' . $search . '%';
        $params['search4'] = '%' . $search . '%';
        $params['search5'] = '%' . $search . '%';
    }
    
    $sql .= ' ORDER BY r.created_at DESC';
    
    // Get total count for pagination
    $countSql = '
        SELECT COUNT(*) as total
        FROM registrations r
        JOIN cities c ON r.city_id = c.id
        JOIN exam_types et ON r.exam_type_id = et.id
        JOIN exam_sessions es ON r.exam_session_id = es.id
        WHERE 1=1
    ';
    
    // Build count params separately to avoid conflicts
    $countParams = [];
    
    // Add same filters to count query
    if ($cityFilter) {
        $countSql .= ' AND r.city_id = :city_id';
        $countParams['city_id'] = $cityFilter;
    }
    if ($examTypeFilter) {
        $countSql .= ' AND r.exam_type_id = :exam_type_id';
        $countParams['exam_type_id'] = $examTypeFilter;
    }
    if ($sessionFilter) {
        $countSql .= ' AND r.exam_session_id = :session_id';
        $countParams['session_id'] = $sessionFilter;
    }
    if ($dateFrom) {
        $countSql .= ' AND r.created_at >= :date_from';
        $countParams['date_from'] = $dateFrom . ' 00:00:00';
    }
    if ($dateTo) {
        $countSql .= ' AND r.created_at <= :date_to';
        $countParams['date_to'] = $dateTo . ' 23:59:59';
    }
    if ($search !== '') {
        $countSql .= ' AND (r.student_first_name LIKE :search1 
                        OR r.student_last_name LIKE :search2 
                        OR r.parent_first_name LIKE :search3 
                        OR r.parent_last_name LIKE :search4
                        OR r.school_name LIKE :search5)';
        $countParams['search1'] = '%' . $search . '%';
        $countParams['search2'] = '%' . $search . '%';
        $countParams['search3'] = '%' . $search . '%';
        $countParams['search4'] = '%' . $search . '%';
        $countParams['search5'] = '%' . $search . '%';
    }
    
    $countStmt = $pdo->prepare($countSql);
    
    try {
        $countStmt->execute($countParams);
    } catch (PDOException $e) {
        die('Count query error: ' . $e->getMessage() . '<br>SQL: ' . $countSql . '<br>Params: ' . print_r($countParams, true));
    }
    
    $totalRecords = $countStmt->fetch()['total'];
    $totalPages = ceil($totalRecords / $perPage);
    
    // Add pagination to main query
    $sql .= ' LIMIT :limit OFFSET :offset';
    $stmt = $pdo->prepare($sql);
    
    // Bind all params
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    
    try {
        $stmt->execute();
    } catch (PDOException $e) {
        die('Query error: ' . $e->getMessage() . '<br>SQL: ' . $sql . '<br>Params: ' . print_r($params, true));
    }
    
    $registrations = $stmt->fetchAll();
    
    // Decrypt phone numbers
    foreach ($registrations as &$reg) {
        try {
            $reg['parent_phone1'] = $encryptionService->decrypt($reg['parent_phone1_encrypted']);
        } catch (Exception $e) {
            $reg['parent_phone1'] = '-';
        }
        
        if ($reg['parent_phone2_encrypted']) {
            try {
                $reg['parent_phone2'] = $encryptionService->decrypt($reg['parent_phone2_encrypted']);
            } catch (Exception $e) {
                $reg['parent_phone2'] = null;
            }
        } else {
            $reg['parent_phone2'] = null;
        }
    }
    unset($reg);
    
    // Get filter options
    $cities = $pdo->query('SELECT * FROM cities WHERE is_active = 1 ORDER BY name')->fetchAll();
    $examTypes = $pdo->query('SELECT * FROM exam_types WHERE is_active = 1 ORDER BY name')->fetchAll();
    
    // Get session info if filtered
    $sessionInfo = null;
    if ($sessionFilter) {
        $stmt = $pdo->prepare('
            SELECT es.*, et.name as exam_type_name, c.name as city_name
            FROM exam_sessions es
            JOIN exam_types et ON es.exam_type_id = et.id
            JOIN cities c ON es.city_id = c.id
            WHERE es.id = :id
        ');
        $stmt->execute(['id' => $sessionFilter]);
        $sessionInfo = $stmt->fetch();
    }
    
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

$pageTitle = 'Kayıtlar';
if ($sessionInfo) {
    $pageTitle .= ' - ' . $sessionInfo['exam_type_name'] . ' (' . $sessionInfo['city_name'] . ', ' . date('d.m.Y H:i', strtotime($sessionInfo['exam_date'] . ' ' . $sessionInfo['exam_time'])) . ')';
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - Uzman Kariyer Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=IBM+Plex+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
    <?php include __DIR__ . '/components/styles.php'; ?>
</head>
<body>
    <?php include __DIR__ . '/components/sidebar.php'; ?>
    
    <div class="main-content">
        <?php include __DIR__ . '/components/header.php'; ?>
        
        <div class="content">
            <!-- Filters -->
            <div class="card mb-3">
                <form method="GET" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px;">
                    <div class="form-group" style="margin-bottom: 0; grid-column: span 2;">
                        <label class="form-label">Öğrenci/Veli/Okul Ara</label>
                        <input type="text" name="search" class="form-input" placeholder="İsim, soyisim veya okul adı..." value="<?= htmlspecialchars($search) ?>">
                    </div>
                    
                    <div class="form-group" style="margin-bottom: 0;">
                        <label class="form-label">Şehir</label>
                        <select name="city" class="form-select">
                            <option value="">Tümü</option>
                            <?php foreach ($cities as $city): ?>
                                <option value="<?= $city['id'] ?>" <?= $cityFilter == $city['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($city['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group" style="margin-bottom: 0;">
                        <label class="form-label">Sınav Türü</label>
                        <select name="exam_type" class="form-select">
                            <option value="">Tümü</option>
                            <?php foreach ($examTypes as $type): ?>
                                <option value="<?= $type['id'] ?>" <?= $examTypeFilter == $type['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($type['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group" style="margin-bottom: 0;">
                        <label class="form-label">Başlangıç Tarihi</label>
                        <input type="date" name="date_from" class="form-input" value="<?= htmlspecialchars($dateFrom) ?>">
                    </div>
                    
                    <div class="form-group" style="margin-bottom: 0;">
                        <label class="form-label">Bitiş Tarihi</label>
                        <input type="date" name="date_to" class="form-input" value="<?= htmlspecialchars($dateTo) ?>">
                    </div>
                    
                    <div style="display: flex; align-items: flex-end; gap: 8px;">
                        <button type="submit" class="btn btn-primary">Filtrele</button>
                        <a href="/admin/registrations" class="btn btn-secondary">Temizle</a>
                    </div>
                </form>
            </div>
            
            <!-- Results -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Kayıtlar (<?= $totalRecords ?> kayıt, Sayfa <?= $page ?>/<?= $totalPages ?>)</h2>
                    <button class="btn btn-secondary btn-sm" onclick="exportToCSV()">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="7 10 12 15 17 10"></polyline>
                            <line x1="12" y1="15" x2="12" y2="3"></line>
                        </svg>
                        Excel'e Aktar
                    </button>
                </div>
                
                <div class="table-container">
                    <table class="table table-compact" id="registrationsTable">
                        <thead>
                            <tr>
                                <th style="width: 40px;">#</th>
                                <th>Öğrenci</th>
                                <th>Sınıf</th>
                                <th>Okul</th>
                                <th>Veli</th>
                                <th>Telefon</th>
                                <th>Sınav</th>
                                <th>Tarih</th>
                                <th style="width: 80px;">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($registrations)): ?>
                                <tr>
                                    <td colspan="9" class="text-center">Kayıt bulunmuyor</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($registrations as $index => $reg): ?>
                                    <tr>
                                        <td style="color: #8d8d8d; font-size: 13px;"><?= $offset + $index + 1 ?></td>
                                        <td>
                                            <strong><?= htmlspecialchars($reg['student_first_name'] . ' ' . $reg['student_last_name']) ?></strong>
                                        </td>
                                        <td><?= $reg['grade_level'] ?><?= $reg['study_field'] ? ' <span style="color: #8d8d8d;">(' . $reg['study_field'] . ')</span>' : '' ?></td>
                                        <td style="max-width: 250px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;" title="<?= htmlspecialchars($reg['school_name']) ?>">
                                            <?= htmlspecialchars($reg['school_name']) ?>
                                        </td>
                                        <td><?= htmlspecialchars($reg['parent_first_name'] . ' ' . $reg['parent_last_name']) ?></td>
                                        <td style="font-family: monospace; font-size: 13px;">
                                            <?= htmlspecialchars($reg['parent_phone1'] ?? '-') ?>
                                            <?php if ($reg['parent_phone2']): ?>
                                                <br><span style="color: #8d8d8d;"><?= htmlspecialchars($reg['parent_phone2']) ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div style="font-size: 13px;">
                                                <strong><?= htmlspecialchars($reg['exam_type_name']) ?></strong><br>
                                                <span style="color: #8d8d8d;"><?= htmlspecialchars($reg['city_name']) ?></span>
                                            </div>
                                        </td>
                                        <td style="font-size: 13px;">
                                            <?= date('d.m.Y', strtotime($reg['exam_date'])) ?><br>
                                            <span style="color: #8d8d8d;"><?= date('H:i', strtotime($reg['exam_time'])) ?></span>
                                        </td>
                                        <td>
                                            <button class="btn btn-ghost btn-sm" onclick="deleteRegistration(<?= $reg['id'] ?>)" title="Sil">
                                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                                    <polyline points="3 6 5 6 21 6"></polyline>
                                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                                    <line x1="10" y1="11" x2="10" y2="17"></line>
                                                    <line x1="14" y1="11" x2="14" y2="17"></line>
                                                </svg>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php
                        $queryParams = $_GET;
                        unset($queryParams['page']);
                        $baseUrl = '/admin/registrations?' . http_build_query($queryParams);
                        $baseUrl .= $queryParams ? '&' : '?';
                        ?>
                        
                        <?php if ($page > 1): ?>
                            <a href="<?= $baseUrl ?>page=<?= $page - 1 ?>" class="pagination-btn">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <polyline points="15 18 9 12 15 6"></polyline>
                                </svg>
                                Önceki
                            </a>
                        <?php endif; ?>
                        
                        <div class="pagination-pages">
                            <?php
                            $start = max(1, $page - 2);
                            $end = min($totalPages, $page + 2);
                            
                            if ($start > 1): ?>
                                <a href="<?= $baseUrl ?>page=1" class="pagination-page">1</a>
                                <?php if ($start > 2): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $start; $i <= $end; $i++): ?>
                                <a href="<?= $baseUrl ?>page=<?= $i ?>" 
                                   class="pagination-page <?= $i === $page ? 'active' : '' ?>">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if ($end < $totalPages): ?>
                                <?php if ($end < $totalPages - 1): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                                <a href="<?= $baseUrl ?>page=<?= $totalPages ?>" class="pagination-page"><?= $totalPages ?></a>
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($page < $totalPages): ?>
                            <a href="<?= $baseUrl ?>page=<?= $page + 1 ?>" class="pagination-btn">
                                Sonraki
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <polyline points="9 18 15 12 9 6"></polyline>
                                </svg>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script>
        async function deleteRegistration(id) {
            if (!confirm('Bu kaydı silmek istediğinizden emin misiniz? Bu işlem geri alınamaz.')) {
                return;
            }
            
            try {
                const response = await fetch('/admin/api/registrations.php', {
                    method: 'POST',
                    headers: { 
                        'Content-Type': 'application/json',
                        'X-HTTP-Method-Override': 'DELETE'
                    },
                    body: JSON.stringify({ id })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    location.reload();
                } else {
                    alert('Hata: ' + (result.error || 'Bilinmeyen hata'));
                }
            } catch (error) {
                alert('Silme işlemi sırasında hata oluştu');
                console.error(error);
            }
        }
        
        function exportToCSV() {
            const table = document.getElementById('registrationsTable');
            let csv = [];
            
            // Headers
            const headers = ['#', 'Öğrenci', 'Sınıf', 'Okul', 'Veli', 'Telefon', 'Sınav', 'Tarih'];
            csv.push(headers.join(','));
            
            // Rows
            table.querySelectorAll('tbody tr').forEach(row => {
                if (row.cells.length < 8) return;
                
                const cells = [
                    row.cells[0].textContent.trim(),
                    '"' + row.cells[1].textContent.trim() + '"',
                    row.cells[2].textContent.trim(),
                    '"' + row.cells[3].textContent.trim() + '"',
                    '"' + row.cells[4].textContent.trim() + '"',
                    row.cells[5].textContent.trim(),
                    '"' + row.cells[6].textContent.trim().replace(/\n/g, ' ') + '"',
                    row.cells[7].textContent.trim().replace(/\n/g, ' ')
                ];
                csv.push(cells.join(','));
            });
            
            // Download
            const blob = new Blob(['\ufeff' + csv.join('\n')], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            link.href = URL.createObjectURL(blob);
            link.download = 'kayitlar_' + new Date().toISOString().split('T')[0] + '.csv';
            link.click();
        }
    </script>
</body>
</html>
