<?php
/**
 * Admin Dashboard
 */

declare(strict_types=1);

require_once __DIR__ . '/components/auth.php';

require_once __DIR__ . '/../vendor/autoload.php';

// Load environment
$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->safeLoad();

// Load database config
$config = require __DIR__ . '/../config/database.php';

try {
    $dsn = sprintf('%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'], $config['host'], $config['port'], 
        $config['database'], $config['charset']
    );
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Get statistics
    $stats = [
        'total_registrations' => $pdo->query('SELECT COUNT(*) FROM registrations')->fetchColumn(),
        'active_sessions' => $pdo->query('SELECT COUNT(*) FROM exam_sessions WHERE is_active = 1 AND exam_date >= CURDATE()')->fetchColumn(),
        'cities_with_exams' => $pdo->query('SELECT COUNT(DISTINCT city_id) FROM exam_sessions WHERE is_active = 1 AND exam_date >= CURDATE()')->fetchColumn(),
        'exam_types' => $pdo->query('SELECT COUNT(*) FROM exam_types WHERE is_active = 1')->fetchColumn(),
    ];
    
    // Recent registrations
    $recentStmt = $pdo->query('
        SELECT r.*, c.name as city_name, et.name as exam_type_name, es.exam_date, es.exam_time
        FROM registrations r
        JOIN cities c ON r.city_id = c.id
        JOIN exam_types et ON r.exam_type_id = et.id
        JOIN exam_sessions es ON r.exam_session_id = es.id
        ORDER BY r.created_at DESC
        LIMIT 10
    ');
    $recentRegistrations = $recentStmt->fetchAll();
    
    // Upcoming exams
    $upcomingStmt = $pdo->query('
        SELECT es.*, et.name as exam_type_name, c.name as city_name,
               COUNT(r.id) as registration_count
        FROM exam_sessions es
        JOIN exam_types et ON es.exam_type_id = et.id
        JOIN cities c ON es.city_id = c.id
        LEFT JOIN registrations r ON es.id = r.exam_session_id
        WHERE es.is_active = 1 AND es.exam_date >= CURDATE()
        GROUP BY es.id
        ORDER BY es.exam_date ASC, es.exam_time ASC
        LIMIT 5
    ');
    $upcomingExams = $upcomingStmt->fetchAll();
    
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

$pageTitle = 'Dashboard';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - Uzman Kariyer Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=IBM+Plex+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
    <?php include __DIR__ . '/components/styles.php'; ?>
</head>
<body>
    <?php include __DIR__ . '/components/sidebar.php'; ?>
    
    <div class="main-content">
        <?php include __DIR__ . '/components/header.php'; ?>
        
        <div class="content">
            <!-- Stats Grid -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-label">Toplam Kayıt</div>
                    <div class="stat-value"><?= number_format($stats['total_registrations']) ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-label">Aktif Sınav Oturumu</div>
                    <div class="stat-value"><?= number_format($stats['active_sessions']) ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-label">Sınav Olan Şehir</div>
                    <div class="stat-value"><?= number_format($stats['cities_with_exams']) ?></div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-label">Sınav Türü</div>
                    <div class="stat-value"><?= number_format($stats['exam_types']) ?></div>
                </div>
            </div>
            
            <!-- Upcoming Exams -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Yaklaşan Sınavlar</h2>
                    <a href="/admin/exam-sessions" class="btn btn-ghost btn-sm">Tümünü Gör</a>
                </div>
                
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Sınav Türü</th>
                                <th>Şehir</th>
                                <th>Tarih</th>
                                <th>Saat</th>
                                <th>Kayıt</th>
                                <th>Kapasite</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($upcomingExams)): ?>
                                <tr>
                                    <td colspan="6" class="text-center">Yaklaşan sınav bulunmuyor</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($upcomingExams as $exam): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($exam['exam_type_name']) ?></td>
                                        <td><?= htmlspecialchars($exam['city_name']) ?></td>
                                        <td><?= date('d.m.Y', strtotime($exam['exam_date'])) ?></td>
                                        <td><?= date('H:i', strtotime($exam['exam_time'])) ?></td>
                                        <td><?= $exam['registration_count'] ?></td>
                                        <td><?= $exam['capacity'] ?? '∞' ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- Recent Registrations -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Son Kayıtlar</h2>
                    <a href="/admin/registrations" class="btn btn-ghost btn-sm">Tümünü Gör</a>
                </div>
                
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Öğrenci</th>
                                <th>Sınav</th>
                                <th>Şehir</th>
                                <th>Sınıf</th>
                                <th>Tarih</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($recentRegistrations)): ?>
                                <tr>
                                    <td colspan="5" class="text-center">Henüz kayıt bulunmuyor</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($recentRegistrations as $reg): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($reg['student_first_name'] . ' ' . $reg['student_last_name']) ?></td>
                                        <td><?= htmlspecialchars($reg['exam_type_name']) ?></td>
                                        <td><?= htmlspecialchars($reg['city_name']) ?></td>
                                        <td><?= $reg['grade_level'] ?>. Sınıf</td>
                                        <td><?= date('d.m.Y H:i', strtotime($reg['created_at'])) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
