<?php
/**
 * Exam Sessions Management
 */

declare(strict_types=1);

require_once __DIR__ . '/components/auth.php';

require_once __DIR__ . '/../vendor/autoload.php';

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__ . '/..');
$dotenv->safeLoad();

$config = require __DIR__ . '/../config/database.php';

try {
    $dsn = sprintf('%s:host=%s;port=%s;dbname=%s;charset=%s',
        $config['driver'], $config['host'], $config['port'], 
        $config['database'], $config['charset']
    );
    $pdo = new PDO($dsn, $config['username'], $config['password'], $config['options']);
    
    // Pagination
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $perPage = 20;
    $offset = ($page - 1) * $perPage;
    
    // Get total count
    $totalRecords = $pdo->query('SELECT COUNT(*) FROM exam_sessions')->fetchColumn();
    $totalPages = ceil($totalRecords / $perPage);
    
    // Get exam sessions with pagination
    $stmt = $pdo->prepare('
        SELECT es.*, et.name as exam_type_name, c.name as city_name,
               COUNT(r.id) as registration_count
        FROM exam_sessions es
        JOIN exam_types et ON es.exam_type_id = et.id
        JOIN cities c ON es.city_id = c.id
        LEFT JOIN registrations r ON es.id = r.exam_session_id
        GROUP BY es.id
        ORDER BY es.exam_date DESC, es.exam_time DESC
        LIMIT :limit OFFSET :offset
    ');
    $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $sessions = $stmt->fetchAll();
    
    // Get exam types for form
    $examTypes = $pdo->query('SELECT * FROM exam_types WHERE is_active = 1 ORDER BY name')->fetchAll();
    
    // Get cities for form
    $cities = $pdo->query('SELECT * FROM cities WHERE is_active = 1 ORDER BY name')->fetchAll();
    
} catch (PDOException $e) {
    die('Database error: ' . $e->getMessage());
}

$pageTitle = 'Sınav Oturumları';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?> - Uzman Kariyer Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=IBM+Plex+Sans:wght@400;500;600;700&display=swap" rel="stylesheet">
    <?php include __DIR__ . '/components/styles.php'; ?>
</head>
<body>
    <?php include __DIR__ . '/components/sidebar.php'; ?>
    
    <div class="main-content">
        <?php include __DIR__ . '/components/header.php'; ?>
        
        <div class="content">
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Sınav Oturumları (<?= $totalRecords ?> kayıt, Sayfa <?= $page ?>/<?= $totalPages ?>)</h2>
                    <button class="btn btn-primary" onclick="openAddModal()">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="12" y1="5" x2="12" y2="19"></line>
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        Yeni Sınav Ekle
                    </button>
                </div>
                
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Sınav Türü</th>
                                <th>Şehir</th>
                                <th>Tarih</th>
                                <th>Saat</th>
                                <th>Konum</th>
                                <th>Kayıt/Kapasite</th>
                                <th>Durum</th>
                                <th>İşlemler</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($sessions)): ?>
                                <tr>
                                    <td colspan="8" class="text-center">Henüz sınav oturumu bulunmuyor</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($sessions as $session): ?>
                                    <?php
                                    $isPast = strtotime($session['exam_date']) < strtotime('today');
                                    $isFull = $session['capacity'] && $session['registration_count'] >= $session['capacity'];
                                    ?>
                                    <tr>
                                        <td><?= htmlspecialchars($session['exam_type_name']) ?></td>
                                        <td><?= htmlspecialchars($session['city_name']) ?></td>
                                        <td><?= date('d.m.Y', strtotime($session['exam_date'])) ?></td>
                                        <td><?= date('H:i', strtotime($session['exam_time'])) ?></td>
                                        <td><?= htmlspecialchars($session['location'] ?? '-') ?></td>
                                        <td>
                                            <?= $session['registration_count'] ?> / <?= $session['capacity'] ?? '∞' ?>
                                            <?php if ($isFull): ?>
                                                <span class="badge badge-warning">Dolu</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!$session['is_active']): ?>
                                                <span class="badge badge-error">Pasif</span>
                                            <?php elseif ($isPast): ?>
                                                <span class="badge badge-info">Geçmiş</span>
                                            <?php else: ?>
                                                <span class="badge badge-success">Aktif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button class="btn btn-ghost btn-sm" onclick="editSession(<?= $session['id'] ?>)">
                                                Düzenle
                                            </button>
                                            <button class="btn btn-ghost btn-sm" onclick="viewRegistrations(<?= $session['id'] ?>)">
                                                Kayıtlar
                                            </button>
                                            <button class="btn btn-ghost btn-sm" onclick="deleteSession(<?= $session['id'] ?>)">
                                                Sil
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php if ($page > 1): ?>
                            <a href="?page=<?= $page - 1 ?>" class="pagination-btn">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <polyline points="15 18 9 12 15 6"></polyline>
                                </svg>
                                Önceki
                            </a>
                        <?php endif; ?>
                        
                        <div class="pagination-pages">
                            <?php
                            $start = max(1, $page - 2);
                            $end = min($totalPages, $page + 2);
                            
                            if ($start > 1): ?>
                                <a href="?page=1" class="pagination-page">1</a>
                                <?php if ($start > 2): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                            <?php endif; ?>
                            
                            <?php for ($i = $start; $i <= $end; $i++): ?>
                                <a href="?page=<?= $i ?>" 
                                   class="pagination-page <?= $i === $page ? 'active' : '' ?>">
                                    <?= $i ?>
                                </a>
                            <?php endfor; ?>
                            
                            <?php if ($end < $totalPages): ?>
                                <?php if ($end < $totalPages - 1): ?>
                                    <span class="pagination-ellipsis">...</span>
                                <?php endif; ?>
                                <a href="?page=<?= $totalPages ?>" class="pagination-page"><?= $totalPages ?></a>
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($page < $totalPages): ?>
                            <a href="?page=<?= $page + 1 ?>" class="pagination-btn">
                                Sonraki
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <polyline points="9 18 15 12 9 6"></polyline>
                                </svg>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Modal -->
    <div class="modal-overlay" id="sessionModal">
        <div class="modal">
            <div class="modal-header">
                <h3 class="modal-title" id="modalTitle">Yeni Sınav Ekle</h3>
                <button class="modal-close" onclick="closeModal()">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <line x1="18" y1="6" x2="6" y2="18"></line>
                        <line x1="6" y1="6" x2="18" y2="18"></line>
                    </svg>
                </button>
            </div>
            
            <form id="sessionForm" onsubmit="return false;">
                <div class="modal-body">
                    <input type="hidden" name="id" id="sessionId">
                    
                    <div class="form-group">
                        <label class="form-label">Sınav Türü *</label>
                        <select name="exam_type_id" id="examTypeId" class="form-select" required>
                            <option value="">Seçiniz</option>
                            <?php foreach ($examTypes as $type): ?>
                                <option value="<?= $type['id'] ?>"><?= htmlspecialchars($type['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Şehir *</label>
                        <select name="city_id" id="cityId" class="form-select" required>
                            <option value="">Seçiniz</option>
                            <?php foreach ($cities as $city): ?>
                                <option value="<?= $city['id'] ?>"><?= htmlspecialchars($city['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Tarih *</label>
                        <input type="date" name="exam_date" id="examDate" class="form-input" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Saat *</label>
                        <input type="time" name="exam_time" id="examTime" class="form-input" required>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Konum</label>
                        <input type="text" name="location" id="location" class="form-input" placeholder="Örn: Ankara Merkez Şube">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Kapasite (Boş bırakılırsa sınırsız)</label>
                        <input type="number" name="capacity" id="capacity" class="form-input" min="1">
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">İptal</button>
                    <button type="submit" class="btn btn-primary">Kaydet</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
        let currentSessionId = null;
        
        function openAddModal() {
            currentSessionId = null;
            document.getElementById('modalTitle').textContent = 'Yeni Sınav Ekle';
            document.getElementById('sessionForm').reset();
            document.getElementById('sessionId').value = '';
            document.getElementById('sessionModal').classList.add('visible');
        }
        
        function closeModal() {
            document.getElementById('sessionModal').classList.remove('visible');
        }
        
        async function editSession(id) {
            try {
                const response = await fetch(`/admin/api/exam-sessions.php?id=${id}`);
                const session = await response.json();
                
                if (session.error) {
                    alert('Hata: ' + session.error);
                    return;
                }
                
                currentSessionId = id;
                document.getElementById('modalTitle').textContent = 'Sınav Düzenle';
                document.getElementById('sessionId').value = session.id;
                document.getElementById('examTypeId').value = session.exam_type_id;
                document.getElementById('cityId').value = session.city_id;
                document.getElementById('examDate').value = session.exam_date;
                document.getElementById('examTime').value = session.exam_time;
                document.getElementById('location').value = session.location || '';
                document.getElementById('capacity').value = session.capacity || '';
                
                document.getElementById('sessionModal').classList.add('visible');
            } catch (error) {
                alert('Sınav bilgileri yüklenirken hata oluştu');
                console.error(error);
            }
        }
        
        function viewRegistrations(sessionId) {
            window.location.href = '/admin/registrations?session_id=' + sessionId;
        }
        
        async function deleteSession(id) {
            if (!confirm('Bu sınav oturumunu silmek istediğinizden emin misiniz?')) {
                return;
            }
            
            try {
                const response = await fetch('/admin/api/exam-sessions.php', {
                    method: 'POST',
                    headers: { 
                        'Content-Type': 'application/json',
                        'X-HTTP-Method-Override': 'DELETE'
                    },
                    body: JSON.stringify({ id })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    location.reload();
                } else {
                    alert('Hata: ' + (result.error || 'Bilinmeyen hata'));
                }
            } catch (error) {
                alert('Silme işlemi sırasında hata oluştu');
                console.error(error);
            }
        }
        
        // Form submit handler
        document.getElementById('sessionForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                exam_type_id: document.getElementById('examTypeId').value,
                city_id: document.getElementById('cityId').value,
                exam_date: document.getElementById('examDate').value,
                exam_time: document.getElementById('examTime').value,
                location: document.getElementById('location').value,
                capacity: document.getElementById('capacity').value,
            };
            
            if (currentSessionId) {
                formData.id = currentSessionId;
            }
            
            try {
                const response = await fetch('/admin/api/exam-sessions.php', {
                    method: 'POST',
                    headers: { 
                        'Content-Type': 'application/json',
                        'X-HTTP-Method-Override': currentSessionId ? 'PUT' : 'POST'
                    },
                    body: JSON.stringify(formData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    alert(result.message);
                    location.reload();
                } else {
                    alert('Hata: ' + (result.error || 'Bilinmeyen hata'));
                }
            } catch (error) {
                alert('Kaydetme işlemi sırasında hata oluştu');
                console.error(error);
            }
        });
        
        // Close modal on overlay click
        document.getElementById('sessionModal').addEventListener('click', function(e) {
            if (e.target === this) closeModal();
        });
    </script>
</body>
</html>
