# Design Document

## Overview

Bu tasarım dokümanı, Aday Öğrenci Kayıt Formu sisteminin teknik mimarisini ve bileşenlerini tanımlar. Sistem, PHP backend, MySQL veritabanı ve TailwindCSS ile mobil öncelikli responsive frontend kullanacaktır. Form, kademeli açılır (progressive disclosure) yaklaşımıyla kullanıcı deneyimini optimize edecektir.

## Architecture

```mermaid
graph TB
    subgraph Frontend
        A[HTML Form] --> B[TailwindCSS Styling]
        A --> C[JavaScript Controller]
        C --> D[Form Validation]
        C --> E[API Client]
    end
    
    subgraph Backend
        F[PHP Router] --> G[Form Controller]
        G --> H[Validation Service]
        G --> I[Database Service]
        G --> J[School API Client]
    end
    
    subgraph Database
        K[(MySQL)]
    end
    
    subgraph External
        L[School API]
    end
    
    E -->|AJAX| F
    I --> K
    J --> L
```

### Katmanlı Mimari

1. **Presentation Layer**: HTML/TailwindCSS form arayüzü
2. **Client Logic Layer**: JavaScript form kontrolü ve validasyon
3. **API Layer**: PHP REST endpoints
4. **Business Logic Layer**: PHP servisler
5. **Data Layer**: MySQL veritabanı

## Components and Interfaces

### Frontend Components

#### 1. FormController (JavaScript)
```javascript
interface FormController {
    init(): void;
    loadCities(): Promise<void>;
    onCityChange(cityId: number): Promise<void>;
    onBranchChange(branchId: number): Promise<void>;
    onGradeLevelChange(level: number): void;
    searchSchools(query: string): Promise<School[]>;
    validateForm(): ValidationResult;
    submitForm(): Promise<SubmitResult>;
}
```

#### 2. APIClient (JavaScript)
```javascript
interface APIClient {
    getCities(): Promise<City[]>;
    getBranches(cityId: number): Promise<Branch[]>;
    getGradeLevels(branchId: number): Promise<GradeLevel[]>;
    searchSchools(cityId: number, query: string): Promise<School[]>;
    submitRegistration(data: RegistrationData): Promise<Response>;
}
```

### Backend Components

#### 1. FormController (PHP)
```php
interface FormControllerInterface {
    public function getCities(): JsonResponse;
    public function getBranches(int $cityId): JsonResponse;
    public function getGradeLevels(int $branchId): JsonResponse;
    public function searchSchools(int $cityId, string $query): JsonResponse;
    public function submitRegistration(Request $request): JsonResponse;
}
```

#### 2. ValidationService (PHP)
```php
interface ValidationServiceInterface {
    public function validateRegistration(array $data): ValidationResult;
    public function sanitizeInput(string $input): string;
    public function validatePhoneNumber(string $phone): bool;
}
```

#### 3. DatabaseService (PHP)
```php
interface DatabaseServiceInterface {
    public function saveRegistration(RegistrationData $data): int;
    public function getCities(): array;
    public function getBranchesByCity(int $cityId): array;
    public function getGradeLevelsByBranch(int $branchId): array;
}
```

### API Endpoints

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/api/cities` | GET | Tüm şehirleri listeler |
| `/api/branches/{cityId}` | GET | Şehre ait şubeleri listeler |
| `/api/grade-levels/{branchId}` | GET | Şubeye ait sınıf düzeylerini listeler |
| `/api/schools/search` | GET | Şehirdeki okulları arar |
| `/api/registration` | POST | Kayıt formunu gönderir |

## Data Models

### City
```php
class City {
    public int $id;
    public string $name;
    public string $code;
    public bool $isActive;
}
```

### Branch
```php
class Branch {
    public int $id;
    public int $cityId;
    public string $name;
    public string $address;
    public bool $isActive;
}
```

### GradeLevel
```php
class GradeLevel {
    public int $id;
    public int $branchId;
    public int $level; // 1-12
    public string $displayName;
    public bool $isActive;
}
```

### StudyField (Alan Bilgisi)
```php
class StudyField {
    public const SAYISAL = 'sayisal';
    public const SOZEL = 'sozel';
    public const ESIT_AGIRLIK = 'esit_agirlik';
    public const DIL = 'dil';
    
    public static function getAll(): array;
    public static function isValidField(string $field): bool;
}
```

### Registration
```php
class Registration {
    public ?int $id;
    public int $cityId;
    public int $branchId;
    public int $gradeLevel;
    public ?string $studyField; // Sadece 10-12. sınıf için
    public string $studentFirstName;
    public string $studentLastName;
    public int $schoolId;
    public string $parentFirstName;
    public string $parentLastName;
    public string $parentPhone1;
    public ?string $parentPhone2;
    public bool $policyAccepted;
    public DateTime $createdAt;
}
```

### School (External API Response)
```php
class School {
    public int $id;
    public string $name;
    public int $cityId;
    public string $type; // İlkokul, Ortaokul, Lise
}
```

### Database Schema

```sql
CREATE TABLE cities (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    code VARCHAR(10) NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE branches (
    id INT PRIMARY KEY AUTO_INCREMENT,
    city_id INT NOT NULL,
    name VARCHAR(200) NOT NULL,
    address TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (city_id) REFERENCES cities(id)
);

CREATE TABLE grade_levels (
    id INT PRIMARY KEY AUTO_INCREMENT,
    branch_id INT NOT NULL,
    level INT NOT NULL CHECK (level BETWEEN 1 AND 12),
    display_name VARCHAR(50) NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (branch_id) REFERENCES branches(id)
);

CREATE TABLE registrations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    city_id INT NOT NULL,
    branch_id INT NOT NULL,
    grade_level INT NOT NULL,
    study_field VARCHAR(20),
    student_first_name VARCHAR(100) NOT NULL,
    student_last_name VARCHAR(100) NOT NULL,
    school_id INT NOT NULL,
    school_name VARCHAR(200) NOT NULL,
    parent_first_name VARCHAR(100) NOT NULL,
    parent_last_name VARCHAR(100) NOT NULL,
    parent_phone1_encrypted VARBINARY(255) NOT NULL,
    parent_phone2_encrypted VARBINARY(255),
    policy_accepted BOOLEAN NOT NULL DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (city_id) REFERENCES cities(id),
    FOREIGN KEY (branch_id) REFERENCES branches(id)
);
```



## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

### Property 1: Branch Filtering by City
*For any* city selection, all branches returned by `getBranches(cityId)` should have their `cityId` field equal to the selected city's ID.
**Validates: Requirements 1.2**

### Property 2: City Change Resets Branch Selection
*For any* form state with a selected branch, when the city selection changes, the branch selection should become null and the new branches should be loaded for the new city.
**Validates: Requirements 1.3**

### Property 3: Grade Level Filtering by Branch
*For any* branch selection, all grade levels returned by `getGradeLevels(branchId)` should have their `branchId` field equal to the selected branch's ID.
**Validates: Requirements 2.1**

### Property 4: Study Field Visibility Based on Grade Level
*For any* grade level selection, the study field input should be visible if and only if the grade level is 10, 11, or 12.
**Validates: Requirements 2.2, 2.3**

### Property 5: School Search Returns City-Filtered Results
*For any* city and search query, all schools returned by `searchSchools(cityId, query)` should have their `cityId` field equal to the selected city's ID and their name should contain the search query.
**Validates: Requirements 3.2**

### Property 6: Required Field Validation
*For any* form submission attempt, if any required field (city, branch, gradeLevel, studentFirstName, studentLastName, schoolId, parentFirstName, parentLastName, parentPhone1, policyAccepted) is empty or null, the validation should return an error for that specific field.
**Validates: Requirements 1.4, 3.3, 3.4, 4.3, 5.2**

### Property 7: Conditional Study Field Validation
*For any* form submission where grade level is 10, 11, or 12, if studyField is null or empty, the validation should return an error. For grade levels 1-9, studyField validation should be skipped.
**Validates: Requirements 2.4**

### Property 8: Phone Number Format Validation
*For any* phone number input, the validation should accept only strings matching the Turkish mobile format (5XX XXX XX XX or 5XXXXXXXXX). Invalid formats should return a validation error.
**Validates: Requirements 4.2, 4.4**

### Property 9: Optional Phone2 Validation
*For any* form submission, if parentPhone2 is empty or null, validation should pass. If parentPhone2 has a value, it should be validated against the phone format rules.
**Validates: Requirements 4.5**

### Property 10: Registration Data Round-Trip
*For any* valid registration data, after saving to database and retrieving, the retrieved data should be equivalent to the original data (excluding auto-generated fields like id and createdAt).
**Validates: Requirements 7.1**

### Property 11: Phone Encryption Round-Trip
*For any* phone number, encrypting and then decrypting should return the original phone number.
**Validates: Requirements 7.4**

### Property 12: Input Sanitization
*For any* string input containing potential XSS or SQL injection patterns, the sanitization function should remove or escape dangerous characters while preserving the safe content.
**Validates: Requirements 7.3**

### Property 13: Loading State During Async Operations
*For any* async operation (API call), the loading state should be true during the operation and false after completion (success or error).
**Validates: Requirements 6.4**

### Property 14: Form State Preservation on Error
*For any* form submission that results in an error, all previously entered form data should remain intact after the error is displayed.
**Validates: Requirements 7.2**

## Error Handling

### Client-Side Errors

| Error Type | Handling Strategy |
|------------|-------------------|
| Validation Error | Inline hata mesajı ilgili alanın altında gösterilir |
| Network Error | Toast notification ile "Bağlantı hatası, lütfen tekrar deneyin" mesajı |
| API Error | Sunucudan gelen hata mesajı kullanıcıya gösterilir |
| Timeout | 30 saniye sonra timeout, yeniden deneme seçeneği sunulur |

### Server-Side Errors

| Error Type | HTTP Code | Response |
|------------|-----------|----------|
| Validation Error | 422 | `{"errors": {"field": ["message"]}}` |
| Not Found | 404 | `{"error": "Resource not found"}` |
| Server Error | 500 | `{"error": "Internal server error"}` |
| Rate Limit | 429 | `{"error": "Too many requests"}` |

### Error Message Display

```javascript
// Hata mesajları Türkçe ve kullanıcı dostu olmalı
const errorMessages = {
    required: '{field} alanı zorunludur',
    invalidPhone: 'Geçerli bir telefon numarası giriniz (5XX XXX XX XX)',
    invalidEmail: 'Geçerli bir e-posta adresi giriniz',
    networkError: 'Bağlantı hatası oluştu, lütfen tekrar deneyin',
    serverError: 'Bir hata oluştu, lütfen daha sonra tekrar deneyin'
};
```

## Testing Strategy

### Unit Testing

PHPUnit kullanılarak backend servisleri test edilecektir:

- **ValidationService Tests**: Her validasyon kuralı için birim testler
- **DatabaseService Tests**: CRUD operasyonları için testler (test database ile)
- **Encryption Tests**: Şifreleme/çözme fonksiyonları için testler

JavaScript tarafında Jest kullanılacaktır:

- **FormController Tests**: Form state yönetimi testleri
- **APIClient Tests**: API çağrıları mock'lanarak test edilecek
- **Validation Tests**: Client-side validasyon testleri

### Property-Based Testing

**PHP**: PHPUnit ile birlikte `eris/eris` veya `giorgiosironi/eris` kütüphanesi kullanılacaktır.

**JavaScript**: `fast-check` kütüphanesi kullanılacaktır.

Her property-based test minimum 100 iterasyon çalıştırılacaktır.

Property testleri şu formatta etiketlenecektir:
```
**Feature: student-registration-form, Property {number}: {property_text}**
```

### Test Categories

1. **Validation Properties** (Property 6, 7, 8, 9)
   - Rastgele form verileri oluşturularak validasyon kuralları test edilecek
   
2. **Data Filtering Properties** (Property 1, 3, 5)
   - Rastgele şehir/şube/okul verileri ile filtreleme doğruluğu test edilecek

3. **State Management Properties** (Property 2, 4, 13, 14)
   - Rastgele kullanıcı etkileşimleri ile state tutarlılığı test edilecek

4. **Round-Trip Properties** (Property 10, 11)
   - Rastgele veriler ile kaydetme/okuma ve şifreleme/çözme tutarlılığı test edilecek

5. **Security Properties** (Property 12)
   - Zararlı input örnekleri ile sanitization test edilecek

### Integration Testing

- Form akışının uçtan uca testi (Cypress veya Playwright)
- API endpoint'lerinin entegrasyon testleri
- Veritabanı işlemlerinin entegrasyon testleri
